# LazySearch plugin for SlimServer by Stuart Hickinbottom 2005
#
# This code is derived from code with the following copyright message:
#
# SlimServer Copyright (c) 2001-2005 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# Allow lazy searching using the SqueezeBox remote control. Note that the
# SlimServer music database must be cleared and rebuilt after installing
# this plugin.
#
# For further details see:
# http://hickinbottom.demon.co.uk/SlimServer/lazy_searching.htm

use strict;

use Slim::Utils::Misc;
use Slim::Utils::Text;
use Slim::Buttons::Search;

package Plugins::LazySearch::Plugin;

# Export the version to the server
use vars qw($VERSION);
$VERSION = "0.1";

use constant LAZYSEARCH_ENABLED_PREFNAME => 'plugin-lazysearch-enabled';

our %functions = (
	'left' => sub {
		my $client = shift;
		Slim::Buttons::Common::popModeRight($client);
	},
	'right' => sub {
		my $client = shift;
		$client->bumpRight();
	},
	'up' => sub {
		my $client = shift;
		$client->bumpUp();
	},
	'down' => sub {
		my $client = shift;
		$client->bumpDown();
	},
	'play' => sub {

		# Toggle the enabled state and tell the user.
		my $client = shift;
		if ( isLazySearchEnabled($client) ) {
			$client->showBriefly(
				{
					'line2' => Slim::Utils::Strings::string(
						'PLUGIN_LAZYSEARCH_DISABLING')
				}
			);

			setLazySearchEnabled( $client, 0 );
		}
		else {
			$client->showBriefly(
				{
					'line2' => Slim::Utils::Strings::string(
						'PLUGIN_LAZYSEARCH_ENABLING')
				}
			);

			setLazySearchEnabled( $client, 1 );
		}
	},
);

# Hash containing replacements for lazy multi-tap searching. Anything not in
# here will just be translated to itself as we don't have any better idea. See
# the lazyMultiTapDecode function for further details of how this works.
my %lazyMultiTapMap = (
	'A' => '2',
	'B' => '22',
	'C' => '222',
	'2' => '2222',
	'D' => '3',
	'E' => '33',
	'F' => '333',
	'3' => '3333',
	'G' => '4',
	'H' => '44',
	'I' => '444',
	'4' => '4444',
	'J' => '5',
	'K' => '55',
	'L' => '555',
	'5' => '5555',
	'M' => '6',
	'N' => '66',
	'O' => '666',
	'6' => '6666',
	'P' => '7',
	'Q' => '77',
	'R' => '777',
	'S' => '7777',
	'7' => '77777',
	'T' => '8',
	'U' => '88',
	'V' => '888',
	'8' => '8888',
	'W' => '9',
	'X' => '99',
	'Y' => '999',
	'Z' => '9999',
	'9' => '99999'
);

# Flag to protect against multiple initialisation or shutdown
my $initialised = 0;

# Below are functions that are part of the standard SlimServer plugin
# interface.

sub lines {
	my $client = shift;
	my $line1  = "";
	my $line2  = "";
	if ( isLazySearchEnabled($client) ) {
		$line1 = Slim::Utils::Strings::string('PLUGIN_LAZYSEARCH_ENABLED1');
		$line2 = Slim::Utils::Strings::string('PLUGIN_LAZYSEARCH_ENABLED2');
	}
	else {
		$line1 = Slim::Utils::Strings::string('PLUGIN_LAZYSEARCH_DISABLED1');
		$line2 = Slim::Utils::Strings::string('PLUGIN_LAZYSEARCH_DISABLED2');
	}

	return ( $line1, $line2 );
}

sub setMode {
	my $client = shift;

	# connect the client's (i.e. the squeezebox) lines to our function
	$client->lines( \&lines );

	# and update them
	$client->update();
}

sub getFunctions {
	return \%functions;
}

sub getDisplayName {
	return 'PLUGIN_LAZYSEARCH';
}

sub webPages {
	my %pages = ( "index\.(?:htm|xml)" => \&handleWebIndex );

	return ( \%pages, "index.html" );
}

sub handleWebIndex {
	my ( $client, $params ) = @_;

	return Slim::Web::HTTP::filltemplatefile( 'plugins/LazySearch/index.html',
		$params );
}

sub initPlugin() {
	return if $initialised;    # don't need to do it twice

	$::d_plugins
	  && Slim::Utils::Misc::msg("LazySearch: Initialising v$VERSION\n");

	# Add hooks to allow filtering of search terms before they are added to
	# the database.
	Slim::DataStores::DBI::DataModel::addPreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_CONTRIBUTOR,
		getDisplayName(), \&lazyPreStoreSearchFilter );
	Slim::DataStores::DBI::DataModel::addPreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_TRACK,
		getDisplayName(), \&lazyPreStoreSearchFilter );
	Slim::DataStores::DBI::DataModel::addPreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_ALBUM,
		getDisplayName(), \&lazyPreStoreSearchFilter );

	# Add hook to allow us to filter search terms to also search for lazy
	# forms
	Slim::Buttons::Search::addSearchTermFilterHook( getDisplayName(),
		\&lazySearchTermFilter );

	# Remember we're now initialised. This prevents multiple-initialisation,
	# which may otherwise cause trouble.
	$initialised = 1;
}

sub shutdownPlugin() {
	return if !$initialised;    # don't need to do it twice

	$::d_plugins && Slim::Utils::Misc::msg("LazySearch: Shutting down\n");

	# Remove our hooks
	Slim::DataStores::DBI::DataModel::removePreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_CONTRIBUTOR,
		getDisplayName() );
	Slim::DataStores::DBI::DataModel::removePreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_TRACK,
		getDisplayName() );
	Slim::DataStores::DBI::DataModel::removePreStoreSearchFilterHook
	  ( Slim::DataStores::DBI::DataModel::SEARCH_COLUMN_ALBUM,
		getDisplayName() );
	Slim::Buttons::Search::removeSearchTermFilterHook( getDisplayName() );

	# We're no longer initialised.
	$initialised = 0;
}

# Below are functions that are specific to this plugin.

sub isLazySearchEnabled($) {
	my $client = shift;

	if (
		!Slim::Utils::Prefs::clientIsDefined(
			$client, LAZYSEARCH_ENABLED_PREFNAME
		)
	  )
	{
		Slim::Utils::Prefs::clientSet( $client, LAZYSEARCH_ENABLED_PREFNAME,
			1 );
	}

	return Slim::Utils::Prefs::clientGet( $client,
		LAZYSEARCH_ENABLED_PREFNAME );
}

sub setLazySearchEnabled($$) {
	my $client  = shift;
	my $enabled = shift;

	Slim::Utils::Prefs::clientSet( $client, LAZYSEARCH_ENABLED_PREFNAME,
		$enabled );
}

# Convert a search string to a lazy-entry encoded search string. This includes
# both the original search term and a lazy-encoded version. Later, when
# searching, both are tried. The original search text is kept in upper-case
# and the lazy version is encoded as digits - the latter is because both the
# original and lazy encoded version is searched in case the user bothers to
# put the search string in properly and this minimises the chance of erroneous
# matching.
#
# called:
#   undef
sub lazyEncode($) {
	my $in_string = shift;
	my $out_string;

	# This translates each searchable character into the number of the key that
	# shares that letter on the remote. Thus, this tells us what keys the user
	# will enter if he doesn't bother to multi-tap to get at the later
	# characters.
	# We do all this on an upper case version, since upper case is all the user
	# can enter through the remote control.
	$out_string = uc $in_string;
	$out_string =~ tr/ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890 /222333444555666777788899991234567890 /;

	# Return a combination of the original search text and a lazy-encoded
	# version.
	return $in_string . "| " . $out_string;
}

# Allow the user to put in lazy searches when adjacent characters are on the
# same key without having to press right arrow or wait a while. It does this by
# translating non-first keys into the right number of the numeric key they
# appear on.
# Decoding "PTNDP" ("STONES") should become "786637"
# and "ODW" ("MONEY") should become "66639".
#
# called:
#   undef
sub lazyMultiTapDecode($) {
	my $in_string  = shift;
	my $out_string = "";

	# Loop through all the characters (back-to-front), and build up an
	# output string with the appropriate replacements.
	while ($in_string) {
		my $in_c  = chop $in_string;
		my $out_c = $lazyMultiTapMap{$in_c};
		$out_c = $in_c unless $out_c;
		$out_string = $out_c . $out_string;
	}

	return $out_string;
}

# Hook function called before a search text form for an artist name
# (contributor), album title or track title is stored in the database during
# music import. This allows us to add on our lazy-encoded version.
sub lazyPreStoreSearchFilter($$) {
	shift;    # We don't care about the type;
	my $searchText = shift;

	# Lazy encode the search string
	return lazyEncode( Slim::Utils::Text::ignoreCaseArticles($searchText) );
}

# Hook function called before a search entered via the remote-control is
# performed. This allows us to also add our lazy-encoded versions.
sub lazySearchTermFilter($$) {
	my $client        = shift;
	my $searchTermRef = shift;

	# Only allow lazy searching if enabled for this client.
	return unless isLazySearchEnabled($client);

	# Loop through the search term array and add lazy versions of each element
	# already in there.
	my $maxTerms = scalar @$searchTermRef;
	for ( my $index = 0 ; $index < $maxTerms ; $index++ ) {
		push @$searchTermRef, lazyMultiTapDecode( @$searchTermRef[$index] );
	}
}

# Standard plugin function to return our message catalogue.
sub strings {
	return '
PLUGIN_LAZYSEARCH
	EN	Lazy Search

PLUGIN_LAZYSEARCH_DESC
	EN	Searching for artists, albums or songs with the Squeezebox remote control usually requires that each remote control button is <i>multi-tapped</i> to enter characters other than the first character marked above the button. For example, the letter "S" requires four presses of the "7" button. This means to search for "SQUEEZE" you have to press the buttons "7777&gt;778833&gt;33999933" (where "&gt;" represents the right arrow button on the remote). Notice also that you have to press the right arrow button on the remote (or wait a short time), between letters that appear on the same button (between "S" and "Q" and also between the two "E"s in the example).</p><p>Lazy searching allows for much faster entry of search text by allowing you to forget about multi-tapping and instead just press each of the remote control buttons once for each of the letters in the text, whatever position the letter appears in above the button. In addition, you no longer need to use the right arrow button or to wait between letters on the same button. So, using lazy searching, entering "SQUEEZE" is reduced to just "7783393".</p><p>The Squeezebox screen will continue to show the letters on the keys that have been pressed, but the search will correctly find what was intended.</p><p>Note that after installing and enabling this plugin a database clear and rescan needs to be performed.

PLUGIN_LAZYSEARCH_ENABLED1
	EN	Lazy searching is currently enabled

PLUGIN_LAZYSEARCH_ENABLED2
	EN	Press PLAY to disable lazy searching

PLUGIN_LAZYSEARCH_DISABLED1
	EN	Lazy searching is currently disabled

PLUGIN_LAZYSEARCH_DISABLED2
	EN	Press PLAY to enable lazy searching

PLUGIN_LAZYSEARCH_ENABLING
	EN	Enabling lazy searching for this player...

PLUGIN_LAZYSEARCH_DISABLING
	EN	Disabling lazy searching for this player...
';
}

1;

__END__

# Local Variables:
# tab-width:4
# indent-tabs-mode:t
# End:
