package Slim::Plugin::PreventStandby::Plugin;

# $Id: Plugin.pm 11021 2006-12-21 22:28:39Z dsully $

# PreventStandby.pm by Julian Neil (julian.neil@internode.on.net)
#
# Prevent the server machine from going into standby when it is streaming
# music to any clients.  Only works in Windows because it uses the CPAN
# Win32:API module.
#
# Excuse my perl.. first time I've ever used it.
#
# Thanks to the PowerSave plugin by Jason Holtzapplefor some basics,
# to various ppl on the slim forums and to CPAN and the Win32::API module.
#
#-> Changelog
#
# 1.0 - 2006-04-05 - Initial Release
#
# 2.0 - 2009-01-03 - Proposed changes by Gordon Harris:
#                    Added "idletime" feature -- waits at least $idletime number
#                    of idle player intervals before allowing standby.  Also, is
#                    "resume aware" -- resets the idle counter on system resume
#                    from standby or hibernation.
#

use strict;

use Slim::Utils::Log;
use Slim::Utils::Prefs;
use Slim::Utils::OSDetect;
require Slim::Plugin::PreventStandby::Settings;


use Win32::API;


# how many seconds between checks for playing clients
my $interval = 60;

# time() we last checked for client activity
my $lastchecktime = 0;

# allowed player idle time intervals.
my $idletime = 0;

# number of intervals that the cliets have been idle.
my $hasbeenidle = 0;

# keep the timer so we can kill it if we want
my $timer = undef;

my $prefs = preferences('plugin.preventstandby');

# Logger object
my $log = Slim::Utils::Log->addLogCategory({
	'category'     => 'plugin.preventstandby',
	'defaultLevel' => 'ERROR',
	'description'  => getDisplayName(),
});

# reference to the windows function of same name
my $SetThreadExecutionState = undef;


sub getFunctions {
	return '';
}

sub getDisplayName {
	return 'PLUGIN_PREVENTSTANDBY';
}

sub getDisplayDescription {
	return "PLUGIN_PREVENTSTANDBY_DESC";
}


sub playersBusy {
	for my $client (Slim::Player::Client::clients()) {
		if ( $client->isUpgrading() || $client->isPlaying() ) {
			$log->debug("Player " . $client->name() . " is busy..\n");
			return 1;
		}
	}
	return 0;
}

sub hasResumed {
	my $currenttime = shift;

	# We've resumed if the current time is more than two minutes later than the last check time, or
	# if the current time is earlier than the last check time (DST time change)
	if ( $currenttime > ($lastchecktime + ($interval * 2)) || $currenttime < $lastchecktime ) {
		$log->debug("System has resumed..\n");
		return 1;
	}
	return 0;
}


sub checkClientActivity {
	my $currenttime = time();
	$timer = undef;

	#$log->debug("Checking Client Activity..\n");

	# Reset the idle countdown counter if 1). scanning, 2). firmware updating or playing, or
	# 4). time-shift (i.e. DST systme time change or we've resumed from standby or hibernation..
	if ($idletime) {
		if ( Slim::Music::Import->stillScanning() || playersBusy() || hasResumed($currenttime) ) {
			$hasbeenidle = 0;
			$log->debug("Resetting idle counter.    " . ($idletime - $hasbeenidle) . " minutes left in allowed idle period.\n");
		} else {
			$hasbeenidle++;
			if ($hasbeenidle < $idletime) {
				$log->debug("Incrementing idle counter. " . ($idletime - $hasbeenidle) . " minutes left in allowed idle period.\n");
			}
		}
	}

	# If idletime is set to zero in settings, ALWAYS prevent standby..
	# Otherwise, only prevent standby if we're still in the idle time-out period..
	if ( (!$idletime) || $hasbeenidle < $idletime) {
			if (defined $SetThreadExecutionState) {
				$log->info("Preventing System Standby..\n");
				$SetThreadExecutionState->Call(1);
			}
	} else {
		$log->info("Players have been idle for $hasbeenidle minutes. Allowing System Standby..\n");
	}

	$lastchecktime = $currenttime;
	startTimer();
	return 1;
}

sub startTimer {
	if (!defined $timer && defined $SetThreadExecutionState) {
		#$log->info("Starting timer..\n");
		$timer = Slim::Utils::Timers::setTimer(undef, time + $interval, \&checkClientActivity);
		if (!defined $timer) {
			$log->error("Starting timer failed!\n");
		}
	}
	return defined($timer);
}

sub stopTimer {
	if (defined($timer)) {
		Slim::Utils::Timers::killSpecific($timer);
		$timer = undef;
	}
}

sub idletime_change {
	my ($pref, $value) = @_;

	$log->debug("Pref $pref changed to $value. Resetting idle counter.\n");
	$idletime = $prefs->get('idletime');

	# Reset our counter on prefs change..
	$hasbeenidle = 0;

	if (!$idletime) {
		$log->debug("System standby now prohibited.\n")
	} else {
		$log->debug("System standby now allowed after $idletime minutes of player idle time.\n")
	}
}


sub initPlugin {
	$log->debug("Init started..\n");
	$idletime = $prefs->get('idletime');
	$hasbeenidle = 0;
	$lastchecktime = time();
	$SetThreadExecutionState = Win32::API->new('kernel32', 'SetThreadExecutionState', 'N', 'N');
	Slim::Plugin::PreventStandby::Settings->new;
	if (!$idletime) {
		$log->debug("System standby now prohibited.\n")
	} else {
		$log->debug("System standby now allowed after $idletime minutes of player idle time.\n")
	}
	$log->debug("Init completed..\n");
	return startTimer();
}


sub shutdownPlugin {
	stopTimer();
}

1;

__END__
