package Plugins::MusicInfoSCR::Settings;

use strict;
use base qw(Slim::Web::Settings);
use Slim::Utils::Prefs;

my $prefs = preferences('plugin.musicinfo');

my $log = Slim::Utils::Log->addLogCategory({
	'category'     => 'plugin.musicinfoscr',
	'defaultLevel' => 'ERROR',
	'description'  => 'PLUGIN_MUSICINFO',
});

sub name {
	return 'PLUGIN_MUSICINFO';
}

sub page {
	return 'plugins/MusicInfoSCR/settings/basic.html';
}

sub needsClient { 1 }

sub validFor {
	my $class = shift;
	my $client = shift;
	
	return !$client->display->isa('Slim::Display::NoDisplay');
}


sub handler {
	my ($class, $client, $params) = @_;

	$params->{'hasDualscreen'} = $client->display->hasScreen2();
	$params->{'isaSqueezebox2'} = $client->display->isa('Slim::Display::Squeezebox2');
	$params->{'canAlternativeVolume'} = ($client->display->vfdmodel() eq Slim::Display::Squeezebox2->vfdmodel());
	$params->{'isGraphical'} = $client->display->isa('Slim::Display::Graphics');

	my @myPrefs = getPrefsList($client, 1);

	if ($params->{'saveSettings'}) {
		for my $pref (@myPrefs) {
			if ($pref eq 'plugin_musicinfo_size_X') {
				$params->{$pref} = $params->{'plugin_musicinfo_size'};
			}

			# font sizes need the appropriate line number added
			if (defined($params->{$pref}) && $pref =~ /A_.*size/) {
				_setFontSize($client, 
					$pref, 
					$params->{$pref}, 
					$params->{'isaSqueezebox2'} ? 'standard' : 'medium',
					'.1');
			}
			elsif (defined($params->{$pref}) && $pref =~ /B_.*size/) {
				_setFontSize($client, 
					$pref, 
					$params->{$pref}, 
					$params->{'isaSqueezebox2'} ? 'standard' : 'medium',
					$params->{$pref} eq 'threeline' ? '.3' : '.2');
			}
			elsif (defined($params->{$pref}) && $pref =~ /_dbl_.*size/) {
				_setFontSize($client, 
					$pref, 
					$params->{$pref}, 
					$params->{'isaSqueezebox2'} ? 'full' : 'large',
					'.2');
			}
			else {
				$prefs->client($client)->set($pref, $params->{$pref});
			}
		}

		updateClientSettings($client);
	}
	
	my $initialized;
	for my $pref (@myPrefs) {
		last if ($initialized = defined($prefs->client($client)->get($pref)));
	}

	Plugins::MusicInfoSCR::Info::init($client);

	if ($params->{'resetSettings'} || !$initialized) {
		initClientSettings($client);
		updateClientSettings($client);
	}

	for my $pref (@myPrefs) {
		$params->{'prefs'}->{$pref} = $prefs->client($client)->get($pref);

		# remove the line number from the font size
		if ($pref =~ /(A|B|_dbl)_.*size/) {
			$params->{'prefs'}->{$pref} =~ s/\.[123]?//;
		}
	}

	$params->{'formatStrings'} = getFormatStrings();
	$params->{'fontList'} = Slim::Web::Settings::Player::Display::getFontOptions($client);

	return $class->SUPER::handler($client, $params);
}

sub _setFontSize {
	my ($client, $pref, $font, $default, $postfix) = @_;
	
	if ($font) {
		$prefs->client($client)->set($pref, $font . $postfix);
	}
	else {
		$prefs->client($client)->set($pref, $default);
	}
}

sub getFormatStrings {
	my %formatStrings;

	my $serverPrefs = preferences('server');
	foreach ( @{$serverPrefs->get('titleFormat')} ) {
		$formatStrings{$_} = $_ if ($_);
	}
	
	# get 3rd party custom items
	my $customItems = getCustomItems();
	foreach (keys %$customItems) {
		$formatStrings{$_} = $_ if ($_);
	}

	# add some special formats...
	$formatStrings{'CURRTIME'} = 'CURRTIME';
	$formatStrings{'LONGDATE'} = 'LONGDATE';
	$formatStrings{'SHORTDATE'} = 'SHORTDATE';
	$formatStrings{'KBPS'} = 'KBPS';
	$formatStrings{'PLAYTIME'} = 'PLAYTIME';
	$formatStrings{'PLAYTIME_PROGRESS'} = 'PLAYTIME_PROGRESS';
	$formatStrings{'SONGTIME'} = 'SONGTIME';
	$formatStrings{'PLAYTIME / SONGTIME'} = 'PLAYTIME / SONGTIME';
	$formatStrings{'PLAYLIST'} = 'PLAYLIST';
	$formatStrings{'PLAYLIST (X_OF_Y)'} = 'PLAYLIST (X_OF_Y)';
	$formatStrings{'NOW_PLAYING'} = 'PLAYING';
	$formatStrings{'NOW_PLAYING (X_OF_Y)'} = 'PLAYING (X_OF_Y)';
	$formatStrings{'X_OF_Y'} = 'X_OF_Y';
	$formatStrings{'X/Y'} = 'X/Y';
	$formatStrings{'PROGRESSBAR'} = 'PROGRESSBAR';
	$formatStrings{''} = 'NOTHING';

	return \%formatStrings; 
}

sub getCustomItems {
	my $customItems;
	
	no strict 'refs';
	my @plugins = Slim::Utils::PluginManager->enabledPlugins();
	
	for my $plugin (@plugins) {
		if(UNIVERSAL::can($plugin,"getMusicInfoSCRCustomItems")) {
			my $items = eval { &{"${plugin}::getMusicInfoSCRCustomItems"}() };
			foreach (keys %{$items}) {
				$log->debug("Got item '$_' for '$plugin'") if ($log->is_debug);
				$customItems->{$_} = $items->{$_};
			}
		}
	}
	use strict 'refs';
	return $customItems;
}

sub getPrefsList {
	my $client = shift;
	my $wantsArray = shift;
	
	# '_?X': second display, '_2X': extended 2nd, '_P?': playlist
	my $myPrefs = {
		plugin_musicinfo_stream_fallback => 1,
		plugin_musicinfo_jump_back => 0,
		plugin_musicinfo_playlist => 0,
		plugin_musicinfo_show_icons => 0,

		plugin_musicinfo_lineA => 'PLAYING (X_OF_Y)',
		plugin_musicinfo_centerA => 'NOTHING',
		plugin_musicinfo_overlayA => 'PLAYTIME_PROGRESS',
		plugin_musicinfo_lineB => 'TITLE (ARTIST)',
		plugin_musicinfo_centerB => 'NOTHING',
		plugin_musicinfo_overlayB => 'NOTHING',
		plugin_musicinfo_line_dbl => 'TITLE (ARTIST)',
		plugin_musicinfo_center_dbl => 'NOTHING',
		plugin_musicinfo_overlay_dbl => 'NOTHING',

		plugin_musicinfo_lineA_P => 'NOTHING',
		plugin_musicinfo_centerA_P => 'NOTHING',
		plugin_musicinfo_overlayA_P => 'NOTHING',
		plugin_musicinfo_lineB_P => 'NOTHING',
		plugin_musicinfo_centerB_P => 'NOTHING',
		plugin_musicinfo_overlayB_P => 'NOTHING',
		plugin_musicinfo_line_dbl_P => 'NOTHING',
		plugin_musicinfo_center_dbl_P => 'NOTHING',
		plugin_musicinfo_overlay_dbl_P => 'NOTHING',
	};

	if ($client->display->isa('Slim::Display::Squeezebox2')) {
		# don't enable alternative volume display for small displays
		$myPrefs->{plugin_musicinfo_alt_volume} = ($client->display->vfdmodel() eq Slim::Display::Squeezebox2->vfdmodel());
		$myPrefs->{plugin_musicinfo_disable_visualizer} = 0;
		$myPrefs->{plugin_musicinfo_show_sleep} = 0;

		$myPrefs->{plugin_musicinfo_size} = 1;
		$myPrefs->{plugin_musicinfo_lineA_size} = '';
		$myPrefs->{plugin_musicinfo_centerA_size} = '';
		$myPrefs->{plugin_musicinfo_overlayA_size} = '';
		$myPrefs->{plugin_musicinfo_lineB_size} = '';
		$myPrefs->{plugin_musicinfo_centerB_size} = '';
		$myPrefs->{plugin_musicinfo_overlayB_size} = '';
		$myPrefs->{plugin_musicinfo_line_dbl_size} = '';
		$myPrefs->{plugin_musicinfo_center_dbl_size} = '';
		$myPrefs->{plugin_musicinfo_overlay_dbl_size} = '';

		$myPrefs->{plugin_musicinfo_size_P} = 1;
		$myPrefs->{plugin_musicinfo_lineA_P_size} = '';
		$myPrefs->{plugin_musicinfo_centerA_P_size} = '';
		$myPrefs->{plugin_musicinfo_overlayA_P_size} = '';
		$myPrefs->{plugin_musicinfo_lineB_P_size} = '';
		$myPrefs->{plugin_musicinfo_centerB_P_size} = '';
		$myPrefs->{plugin_musicinfo_overlayB_P_size} = '';
		$myPrefs->{plugin_musicinfo_line_dbl_P_size} = '';
		$myPrefs->{plugin_musicinfo_center_dbl_P_size} = '';
		$myPrefs->{plugin_musicinfo_overlay_dbl_P_size} = '';
	}
	
	if ($client->display->isa('Slim::Display::Transporter')) {
		$myPrefs->{plugin_musicinfo_extended} = 0;

		$myPrefs->{plugin_musicinfo_size_X} = 1;
		$myPrefs->{plugin_musicinfo_lineB} = 'DISC-TRACKNUM. TITLE';
		$myPrefs->{plugin_musicinfo_overlayB} = 'NOTESYMBOL';
		$myPrefs->{plugin_musicinfo_lineA_X} = 'ALBUM';
		$myPrefs->{plugin_musicinfo_centerA_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayA_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_lineB_X} = 'ARTIST';
		$myPrefs->{plugin_musicinfo_centerB_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayB_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_line_dbl_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_center_dbl_X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlay_dbl_X} = 'NOTHING';

		$myPrefs->{plugin_musicinfo_lineA_X_size} = '';
		$myPrefs->{plugin_musicinfo_centerA_X_size} = '';
		$myPrefs->{plugin_musicinfo_overlayA_X_size} = '';
		$myPrefs->{plugin_musicinfo_lineB_X_size} = '';
		$myPrefs->{plugin_musicinfo_centerB_X_size} = '';
		$myPrefs->{plugin_musicinfo_overlayB_X_size} = '';
		$myPrefs->{plugin_musicinfo_line_dbl_X_size} = '';
		$myPrefs->{plugin_musicinfo_center_dbl_X_size} = '';
		$myPrefs->{plugin_musicinfo_overlay_dbl_X_size} = '';

		$myPrefs->{plugin_musicinfo_size_2X} = 1;
		$myPrefs->{plugin_musicinfo_lineA_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_centerA_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayA_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_lineB_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_centerB_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayB_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_line_dbl_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_center_dbl_2X} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlay_dbl_2X} = 'NOTHING';

		$myPrefs->{plugin_musicinfo_lineA_2X_size} = '';
		$myPrefs->{plugin_musicinfo_centerA_2X_size} = '';
		$myPrefs->{plugin_musicinfo_overlayA_2X_size} = '';
		$myPrefs->{plugin_musicinfo_lineB_2X_size} = '';
		$myPrefs->{plugin_musicinfo_centerB_2X_size} = '';
		$myPrefs->{plugin_musicinfo_overlayB_2X_size} = '';
		$myPrefs->{plugin_musicinfo_line_dbl_2X_size} = '';
		$myPrefs->{plugin_musicinfo_center_dbl_2X_size} = '';
		$myPrefs->{plugin_musicinfo_overlay_dbl_2X_size} = '';

		$myPrefs->{plugin_musicinfo_size_PX} = 1;
		$myPrefs->{plugin_musicinfo_lineA_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_centerA_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayA_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_lineB_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_centerB_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlayB_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_line_dbl_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_center_dbl_PX} = 'NOTHING';
		$myPrefs->{plugin_musicinfo_overlay_dbl_PX} = 'NOTHING';

		$myPrefs->{plugin_musicinfo_lineA_PX_size} = '';
		$myPrefs->{plugin_musicinfo_centerA_PX_size} = '';
		$myPrefs->{plugin_musicinfo_overlayA_PX_size} = '';
		$myPrefs->{plugin_musicinfo_lineB_PX_size} = '';
		$myPrefs->{plugin_musicinfo_centerB_PX_size} = '';
		$myPrefs->{plugin_musicinfo_overlayB_PX_size} = '';
		$myPrefs->{plugin_musicinfo_line_dbl_PX_size} = '';
		$myPrefs->{plugin_musicinfo_center_dbl_PX_size} = '';
		$myPrefs->{plugin_musicinfo_overlay_dbl_PX_size} = '';
	}
	
	return ($wantsArray ? keys %{$myPrefs} : $myPrefs);
}

sub updateClientSettings {
	my $client = shift;

	# hack to get round being called by notify and web setup - find the client..
	$client = $client->client unless ($client->isa('Slim::Player::Client'));

	return unless ($client && $client->isa('Slim::Player::Player'));

	# alternative Now Playing display
	if ($prefs->client($client)->get('plugin_musicinfo_playlist')) {
		$client->customPlaylistLines(\&Plugins::MusicInfoSCR::Plugin::playlistMusicInfoLines);
		$client->suppressStatus(1);
	} else {
		$client->customPlaylistLines(undef);
		$client->suppressStatus(undef);
	}

	# alternative volume screen - SB2+ only
	if ($client->display->vfdmodel() eq Slim::Display::Squeezebox2->vfdmodel()) {
		$client->customVolumeLines(
			$prefs->client($client)->get('plugin_musicinfo_alt_volume') ? \&Plugins::MusicInfoSCR::Plugin::volumeLines : undef
		);
	}

	# alternative second info screen - Transporter only
	if ($client->display->isa('Slim::Display::Transporter')) {
		$client->lines2periodic(
			$prefs->client($client)->get('plugin_musicinfo_extended') 
				? \&Plugins::MusicInfoSCR::Plugin::extendedTextModeLines
				: \&Slim::Player::Player::currentSongLines
		);
	}
	
	# initialise the display cache
	Plugins::MusicInfoSCR::Info::initDisplayCache($client);
	Plugins::MusicInfoSCR::Info::init($client);
}

sub initClientSettings {
	my $client = shift;
	return if (not defined $client);

	my $defaults = getPrefsList($client);
	foreach (keys %$defaults) {
		$prefs->client($client)->set($_, $defaults->{$_}); 
	}
}


1;

__END__
