# MusicInfoSCR::Plugin.pm by mh 2005-2006, parts by kdf Dec 2003, fcm4711 Oct 2005, triode Sept 2006
#
# This code is derived from code with the following copyright message:
#
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.
#
# Changelog
# 4.1  - add sleep buttons
# 4.00 - SlimServer 7.0 compliance
#      - WYSIWYG editor for the settings
#      - rely on server side caching of contents, don't cache locally (and therefore remove whitelist as well)
#      - code cleanup
# 3.05 - add "X/Y" format
# 3.04 - add BUFFERBAR format
# 3.03 - add new BUFFER format
# 3.02 - add new NOTESYMBOL format
# 3.01 - add new KBPS format (thanks schiegl http://forums.slimdevices.com/showpost.php?p=162830&postcount=4)
# 3.00 - renamed to "Music Information Screen"
#      - improve icon handling (use font files)
#      - add alternative volume, playlist and extended text screen - thanks to Triode
#      - new tags PLAYTIME_PROGRESS, SONGTIME, DATE et al.
#      - move settings to their own page
#      - add format whitelist: entries will _always_ be updated, never cached
#      - add option to fix a display's font size
#      - give up on slimserver <6.5: most new features are 6.5 only
#      - add a reset button, fix initialization
# 2.44- improved internet radio station handling
# 2.43- improved 6.5 compatibility
# 2.42- fix an issue with playtime
#     - only display second screen when in extended text mode
# 2.41- optionally disable visualizer
#     - fix some ugly copy/paste stuff
# 2.40- add support for Transporter dual screen
#     - _really_ emove that support for 3rd party plugins... (see 2.30)
# 2.34- added Dutch translation - thanks a lot Patrick!
# 2.33- remove progress bar from PLAYTIME in double size font mode
# 2.32- q'n'd fix for the progressbar in double size font crash...
# 2.31- use standard text for screensaver activation
# 2.30- remove that support again, as addFormat() should do the same...
# 2.20- add support for third party plugins (thanks a lot Erland Isaksson!)
# 2.10- re-unite versions for 6.2.x and 6.5
#     - try to better handle button presses when jumping to the playlist
# 2.09- add LONGDATE/SHORTDATE variables - thanks Urs Zurbuchen
# 2.07- add Italian translation - thanks Riccardo!
# 2.05- fix an issue when radio stations would display a blank value for tags like "TITLE (ARTIST)"
# 2.02- prepare plugin for use with slimserver 6.5
# 2.01- fixed some issues with too aggressive caching and added one pixel before the icons
# 2.0 - add icons for shuffle and repeat modes - thanks to Felix Müller
#     - remove 6.1 stuff
# 1.9 - prepare for new %lines hash and prefs methods (>=6.2)
# 1.85- James Craig added a fullscreen progress bar - thanks, James!
# 1.8 - added support for new scrolling code in 6.1
# 1.73- fixed display change after scrolling of second line
# 1.72- fixed PLAYTIME display for streams that display 11329320:33:39...
# 1.71- fixed CURRTIME display (don't cache the current time...)
# 1.7 - removed manual update of line2 (only with >=6.1)
# 1.61- don't translate format strings (eg. X_OF_Y)
# 1.6 - improved PLAYTIME progressbar with different displays
# 1.51- improved online stream handling
# 1.5 - fixes for SlimServer v6.0
#     - added display cache for much lower cpu load
# 1.4 - added "CURRENTTIME", "PLAYLIST"
#     - added option to display playlist name if ALBUM/ARTIST is not available (streams)
# 1.3 - added general "FORMAT (X_OF_Y)"
#     - some cosmetics with PLAYTIME (put a space before progressbar, if reasonable)
# 1.2 - fixed progressbar in PLAYTIME
#     - added "X out of Y" style
#     - added french translation
# 1.1 - handle empty playlists, stopped, paused
#     - added jump back configuration: jump back to where you left or go to playlist
#     - added second value (right) for double line
# 1.0 - initial release

package Plugins::MusicInfoSCR::Plugin;

use strict;

use Plugins::MusicInfoSCR::Settings;
use Plugins::MusicInfoSCR::Info;
use Plugins::MusicInfoSCR::Web;
use Slim::Utils::Strings qw (string);
use Slim::Utils::Prefs;

my $log = Slim::Utils::Log->addLogCategory({
	'category'     => 'plugin.musicinfoscr',
	'defaultLevel' => 'ERROR',
	'description'  => 'PLUGIN_MUSICINFO',
});

use vars qw($VERSION);
$VERSION = substr( q$Revision: 4.1.3 $, 10 );

my $prefs = preferences('plugin.musicinfo');

sub getDisplayName {
	return 'PLUGIN_MUSICINFO';
}

sub initPlugin {
	Slim::Control::Request::subscribe(
		\&Plugins::MusicInfoSCR::Settings::updateClientSettings, 
		[['client'], ['new']]
	);
	
	Plugins::MusicInfoSCR::Settings->new();

	Slim::Buttons::Common::addSaver( 'SCREENSAVER.musicinfo',
		getScreensaverMusicInfo(),
		\&setScreensaverMusicInfoMode,
		\&leaveScreensaverMusicInfoMode,
		'PLUGIN_MUSICINFO');

	Slim::Web::HTTP::addPageFunction("plugins/MusicInfoSCR/settings/preview.html", 
		\&Plugins::MusicInfoSCR::Web::handlePreview);

	Slim::Music::TitleFormatter::addFormat('SONGTIME', \&Plugins::MusicInfoSCR::Info::getSongTime);
	Slim::Music::TitleFormatter::addFormat('KBPS', \&Plugins::MusicInfoSCR::Info::getKBPS);
	Slim::Music::TitleFormatter::addFormat('NOW_PLAYING', sub { return string('PLAYING'); });
	Slim::Music::TitleFormatter::addFormat('PLAYING', sub { return string('PLAYING'); });
}

my %oldVisualizer = ();

sub lines {
	my $client = shift;

	my $saver = Slim::Player::Source::playmode($client) eq 'play' ? 'screensaver' : 'idlesaver';
	
	return {
		'line1' => $client->string('PLUGIN_MUSICINFO'),
		'line2' => $client->string('SETUP_SCREENSAVER_USE'),
		'overlay2' => Slim::Buttons::Common::checkBoxOverlay($client, $prefs->client($client)->get($saver) eq 'SCREENSAVER.musicinfo')
	};
}

# entirely stolen from Triode's demo code :-)
sub volumeLines {
	my $client = shift;
	my $vol = shift;

	# custom character names are not defined as entire screen components use the volume font
	# uses font file volumeSB2.2 for volume graphic and text:
	#  chr 1 - space before start of graphic bar
	#  chr 2 - 'dB' icon
	#  chr 3 - infinity character
	#  chr 65 - 65+32 - graphic bars
	#  numeric characters are used as normal from this font

	if (!defined $vol->{value} && $client->modeParam('valueRef')) {
		$vol = ${$client->modeParam('valueRef')};
	}

	my $dbVal = $vol ? sprintf('%.1f', -abs(($vol / 2) - 50)) : '-' . chr(3);
	$dbVal .= chr(2);

	my $bar = chr(1);

	for (my $i = 0; $i <= 32; $i++) {
		last if ($i > $vol/3);
		$bar .= chr($i+65);
	}

	return {
		'line'    => [ $client->string('VOLUME'), $bar ],
		'overlay' => [ undef, $dbVal ],
		'fonts' => {
			'graphic-320x32' => { 'line' => [ 'standard.1', 'MIScrVolumeSB2.2' ], 'overlay' => [ undef, 'MIScrVolumeSB2.2' ] },
		}
	};
}

my %screensaverMusicInfoFunctions = (
	'done' => sub {
		my ( $client, $funct, $functarg ) = @_;
		Slim::Buttons::Common::popMode($client);
  		if (not $prefs->client($client)->get('plugin_musicinfo_jump_back')) {	
			if (Slim::Buttons::Common::mode($client) eq 'playlist') {
				Slim::Buttons::Common::popMode($client);
			}
			Slim::Buttons::Common::pushMode( $client, 'playlist' );
			Slim::Hardware::IR::resendButton($client);
  		}
		$client->update();

		# pass along ir code to new mode if requested
		if ( defined $functarg && $functarg eq 'passback' ) {
			Slim::Hardware::IR::resendButton($client);
		}
	}
);

sub getScreensaverMusicInfo {
	return \%screensaverMusicInfoFunctions;
}

sub leaveScreensaverMusicInfoMode {
	my $client = shift;

	$client->modeParam('visu', $oldVisualizer{$client});
	$client->update({'screen2' => {}});
}

sub setScreensaverMusicInfoMode {
	my $client = shift;

	# setting this param will call client->update() frequently
	$client->modeParam('modeUpdateInterval', 1);

	Plugins::MusicInfoSCR::Info::init($client);

	# optionally disable visualizer
	$oldVisualizer{$client} = $client->modeParam('visu');
	$client->modeParam('visu', [0]) if ($prefs->client($client)->get('plugin_musicinfo_disable_visualizer'));

	if ($client->display->isa('Slim::Display::Transporter')) {
		$client->modeParam('screen2', 'MusicInfoSCR');
		$client->lines( \&screensaverMusicInfoLinesDual );
	}
	else {
		$client->lines( \&screensaverMusicInfoLines );
	}
}

sub playlistMusicInfoLines {
	my $client = shift;

	return if (!defined $client);

	my $infoLines;

	my $tot = Slim::Player::Playlist::count($client);

	if ($tot < 1) {
		$infoLines = {
			'line' => [ $client->string('NOW_PLAYING'), $client->string('NOTHING') ]
		};
	}

	else {
		my $songIndex = Slim::Buttons::Playlist::browseplaylistindex($client);

		if ( $songIndex >= $tot ) {
			$songIndex = Slim::Buttons::Playlist::browseplaylistindex($client, $tot - 1);
		}

		# see whether we copy the settings from the screensaver or not
		$infoLines = Plugins::MusicInfoSCR::Info::getMusicInfoLines(
			$client, 
			($prefs->client($client)->get('plugin_musicinfo_playlist') == 2) ? '_P' : '',
			$songIndex
		);

		$infoLines->{'screen2'} = Plugins::MusicInfoSCR::Info::getMusicInfoLines(
			$client, 
			($prefs->client($client)->get('plugin_musicinfo_playlist') == 2) ? '_PX' : '_X',
			$songIndex
		) if ($client->display->showExtendedText());
	}

	return $infoLines;
}

sub extendedTextModeLines {
	my $client = shift;

	# see whether we copy the settings from the screensaver or not
	my $screen = screensaverMusicInfoLines($client, undef, 
		$prefs->client($client)->get('plugin_musicinfo_extended') == 2 ? '_2X' : '');
	
	return { 'screen2' => $screen }; 
}

sub screensaverMusicInfoLinesDual {
	my $client = shift;
	my $infoLines = screensaverMusicInfoLines($client);
	$infoLines->{'screen2'} = screensaverMusicInfoLines($client, undef, '_X') if ($client->display->showExtendedText());
	return $infoLines;
}

sub screensaverMusicInfoLines {
	my ($client, $args, $screen) = @_;
	return Plugins::MusicInfoSCR::Info::getMusicInfoLines($client, $screen);
}

sub getMusicInfoSCRCustomItems {
	return {
		'BUFFER' => {
			'cb' => \&Plugins::MusicInfoSCR::Info::getBufferFullness,
			'cache' => 1	# don't update more than once a second...
		},
		'NOTESYMBOL' => {
			'cb' => sub {
					my $client = shift;
					return $client->symbols('notesymbol');
			}
		}
	};
}

1;
