# Plugins::MusicInfoSCR::Info.pm by mh 2006/7
#
# This code is derived from code with the following copyright message:
#
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.

use strict;

package Plugins::MusicInfoSCR::Info;

use Slim::Utils::Strings qw (string);
use Slim::Utils::Log;
use Slim::Utils::Prefs;

my $prefs = preferences('plugin.musicinfo');
my $serverPrefs = preferences('server');

my $log = Slim::Utils::Log->addLogCategory({
	'category'     => 'plugin.musicinfoscr',
	'defaultLevel' => 'ERROR',
	'description'  => 'PLUGIN_MUSICINFO',
});

my %displayCache = ();
my $customItems = ();

sub init {
	my $client = shift;

	$customItems = Plugins::MusicInfoSCR::Settings::getCustomItems() unless ($customItems);

	if (!defined $prefs->client($client)->get('plugin_musicinfo_stream_fallback')) {	
		Plugins::MusicInfoSCR::Settings::initClientSettings($client);
	}
}

sub initDisplayCache {
	%displayCache = undef;
}

sub initDisplayModes {
	my $client = shift;
	my $postfix = shift;

	return if ($displayCache{$client}->{'model'} 
		&& defined $displayCache{$client}->{'graphical'} 
		&& $displayCache{$client}->{'postfix'} eq $postfix);

	if ($client->isa('Slim::Player::Squeezebox2')) {
		$displayCache{$client}->{'postfix'} = $postfix;
		$displayCache{$client}->{'model'} = 'SB2';
		$displayCache{$client}->{'graphical'} = 1;
		$displayCache{$client}->{'font'} =
			($prefs->client($client)->get('plugin_musicinfo_size' . $postfix))
				? undef
				: { $client->display->vfdmodel() => {
						'line' => [ 
							$prefs->client($client)->get("plugin_musicinfo_lineA$postfix" . "_size"), 
							$prefs->client($client)->get("plugin_musicinfo_line_dbl$postfix" . "_size"), 
							$prefs->client($client)->get("plugin_musicinfo_lineB$postfix" . "_size")
						],
						'center' => [
							$prefs->client($client)->get("plugin_musicinfo_centerA$postfix" . "_size"), 
							$prefs->client($client)->get("plugin_musicinfo_center_dbl$postfix" . "_size"), 
							$prefs->client($client)->get("plugin_musicinfo_centerB$postfix" . "_size")
						],
						'overlay' => [
							$prefs->client($client)->get("plugin_musicinfo_overlayA$postfix" . "_size"),
							$prefs->client($client)->get("plugin_musicinfo_overlay_dbl$postfix" . "_size"),
							$prefs->client($client)->get("plugin_musicinfo_overlayB$postfix" . "_size")
						],
					}
				};
	}
	elsif ($client->isa('Slim::Player::Squeezebox') && $client->display->isa('Slim::Display::SqueezeboxG')) {
		$displayCache{$client}->{'model'} = 'SBG';
		$displayCache{$client}->{'graphical'} = 1;
	}
}

sub getMusicInfoLines {
	my ($client, $postfix, $songIndex) = @_;

	$postfix ||= '';

	my ( $line1, $center1, $overlay1, $line2, $center2, $overlay2, $line3, $center3, $overlay3 ) 
			= ('', '', '', '', '', '', '', '', '');

	my $song;

	my $nowPlaying = (!defined $songIndex || Slim::Player::Source::playingSongIndex($client) == $songIndex);

	# single line mode: when large font, and font not overwritten
	my $singleLine = ($client->linesPerScreen() == 1 && $prefs->client($client)->get("plugin_musicinfo_size$postfix"));

	initDisplayModes($client, $postfix);

	# if there's nothing playing just display... nothing!
	if (Slim::Player::Playlist::count($client) < 1 && $postfix ne '_X') {
		$line1 = string('NOW_PLAYING');
		$line2 = $line3 = string('NOTHING');
	}
	else {
		$song = Slim::Player::Playlist::song($client, $songIndex);

		if ($singleLine) {
			$line2 = getFormatString($client, "plugin_musicinfo_line_dbl$postfix" || "plugin_musicinfo_lineB$postfix", $song, $songIndex);
			$center2 = getFormatString($client, "plugin_musicinfo_center_dbl$postfix", $song, $songIndex);
			$overlay2 = getFormatString($client, "plugin_musicinfo_overlay_dbl$postfix", $song, $songIndex);
		}
		else {
			$line1 = getFormatString($client, "plugin_musicinfo_lineA$postfix", $song, $songIndex);
			$center1 = getFormatString($client, "plugin_musicinfo_centerA$postfix", $song, $songIndex);
			$overlay1 = getFormatString($client, "plugin_musicinfo_overlayA$postfix", $song, $songIndex);
			
			if ($client->isa('Slim::Player::Squeezebox2')) {
				$line2 = getFormatString($client, "plugin_musicinfo_lineB$postfix", $song, $songIndex);
				$center2 = getFormatString($client, "plugin_musicinfo_centerB$postfix", $song, $songIndex);
				$overlay2 = getFormatString($client, "plugin_musicinfo_overlayB$postfix", $song, $songIndex);

				if (!$prefs->client($client)->get('plugin_musicinfo_size' . $postfix)) {
					$line3 = $line2;
					$center3 = $center2;
					$overlay3 = $overlay2;

					$line2 = getFormatString($client, "plugin_musicinfo_line_dbl$postfix", $song, $songIndex);
					$center2 = getFormatString($client, "plugin_musicinfo_center_dbl$postfix", $song, $songIndex);
					$overlay2 = getFormatString($client, "plugin_musicinfo_overlay_dbl$postfix", $song, $songIndex);					
				}
			}
			else {
				# pre-SB2 players can't handle three lines - use third as second...
				$line2 = getFormatString($client, "plugin_musicinfo_lineB$postfix", $song, $songIndex);
				$center2 = getFormatString($client, "plugin_musicinfo_centerB$postfix", $song, $songIndex);
				$overlay2 = getFormatString($client, "plugin_musicinfo_overlayB$postfix", $song, $songIndex);
			}

			# if there's no information for the first line, display the playlistname or "now playing..." 
			# (eg. for author information with online streaming etc.)
			if ($postfix ne '_X' && $prefs->client($client)->get("plugin_musicinfo_lineA$postfix") and not $line1) {
				$line1 = getFormatString($client, 'PLAYLIST', $song, $songIndex);
				$line1 = string('NOW_PLAYING') if ($line1 eq $line2 || not $line1);
			}
		}
	}

	# some special information (on first display only, not in full size mode)
	if ($postfix !~ /X$/ && !$singleLine) {
		# handle special modes... 
		if (Slim::Player::Source::playmode($client) eq 'pause' && $nowPlaying) {
			$line1 = $client->string('PAUSED') . ($line1 ? ' - ' . $line1 : '');
		}
		elsif (Slim::Player::Source::playmode($client) eq 'stop' && $nowPlaying) {
			$line1 = $client->string('STOPPED') . ($line1 ? ' - ' . $line1 : '');
		}

		# Repeat/Shuffle icons - right; don't display icons in full size mode
		if (my $showIcons = $prefs->client($client)->get('plugin_musicinfo_show_icons')) {
			$overlay1 = buttonIcons($client, $showIcons, $overlay1) if ($showIcons =~ /^TOPRIGHT[2]*$/);
		}

		# sleep icons - right
		if (my $showIcons = $prefs->client($client)->get('plugin_musicinfo_show_sleep')) {
			$overlay1 = sleepIcons($client, $showIcons, $overlay1) if ($showIcons =~ /^TOPRIGHT[23]*$/);
		}
	}

	# PLAYTIME stuff...
	$line1 = getPlayTime($client, $line1, $overlay1, 1, $song, $nowPlaying);
	$line2 = getPlayTime($client, $line2, $overlay2, 2, $song, $nowPlaying);
	$line3 = getPlayTime($client, $line3, $overlay3, 2, $song, $nowPlaying);

	# Repeat/Shuffle icons - left
	if ($postfix !~ /X$/ && !$singleLine) {
		if (my $showIcons = $prefs->client($client)->get('plugin_musicinfo_show_icons')) {
			$line1 = buttonIcons($client, $showIcons, $line1) if ($showIcons =~ /^TOPLEFT[2]*$/);
		}

		# sleep icons - right
		if (my $showIcons = $prefs->client($client)->get('plugin_musicinfo_show_sleep')) {
			$line1 = sleepIcons($client, $showIcons, $line1) if ($showIcons =~ /^TOPLEFT[23]*$/);
		}
	}
	
	$overlay1 = getPlayTime($client, $overlay1, $line1, 1, $song, $nowPlaying);
	$overlay2 = getPlayTime($client, $overlay2, $line2, 2, $song, $nowPlaying);
	$overlay3 = getPlayTime($client, $overlay3, $line3, 2, $song, $nowPlaying);

	return {
		'line' => [ $line1, $line2, $line3 ],
		'center' => [ $center1 || undef, $center2 || undef, $center3 || undef ],
		'overlay' => [ $overlay1, $overlay2, $overlay3 ],
		'fonts' => $displayCache{$client}->{'font'}
	};
}

sub getFormatString {
	my ($client, $formatString, $song, $songIndex) = @_;

	# get the real format definition if we're passed a pref identifier
	$formatString = $prefs->client($client)->get($formatString) if ($formatString =~ /^plugin_musicinfo_/);

	my $formattedString = $formatString;
	my $titleOnly;

	my $albumOrArtist = ($formatString =~ /(ALBUM|ARTIST)/i);
	my $string;
			
	# Playlistname
	if ($formatString =~ /PLAYLIST/) {
		if ($string = $client->currentPlaylist()) {
			$string = Slim::Music::Info::standardTitle($client, $string);
		}
		else {
			$string = '';
		}
		$formattedString =~ s/PLAYLIST/$string/g;
	}
	
	# Song counter
	if ($formattedString =~ /X_OF_Y/) {
		$string = sprintf("%d %s %d", 
				(defined $songIndex ? $songIndex : Slim::Player::Source::playingSongIndex($client)) + 1, 
				$client->string('OUT_OF'), Slim::Player::Playlist::count($client));
		$formattedString =~ s/X_OF_Y/$string/g;
	}
	elsif ($formattedString =~ /X\/Y/) {
		$string = sprintf("%d/%d", 
				(defined $songIndex ? $songIndex : Slim::Player::Source::playingSongIndex($client)) + 1, 
				Slim::Player::Playlist::count($client));
		$formattedString =~ s/X\/Y/$string/g;
	}

	$formattedString = getCustomItem($client, $song, $formattedString);

	# Title: as infoFormat() seems to give us some problems with internet radio stations, we'll handle this manually...
	if (($formatString =~ /TITLE/) && Slim::Music::Info::isRemoteURL($song->url)) {
		$titleOnly = Slim::Music::Info::getCurrentTitle($client, $song->url);
		$formattedString =~ s/TITLE/_MH_TMP_TITLE_/g
	}

	$formattedString = Slim::Music::Info::displayText($client, $song, $formattedString);

	$formattedString =~ s/_MH_TMP_TITLE_/$titleOnly/g;

	# one more special case... ARTIST with a radio stream results in... nothing. Replace by TITLE		
	if ((!$formattedString) && ($formatString =~ /TITLE/) && $titleOnly) {
		$formattedString = $titleOnly;
	}
	
	# if ARTIST/ALBUM etc. is empty, replace them by some "Now playing..."
	if ($albumOrArtist) {
		my $tmpFormattedString = $formattedString;
		my $noArtist = $client->string('NO_ARTIST');
		my $noAlbum = $client->string('NO_ALBUM');
		$tmpFormattedString =~ s/($noAlbum|$noArtist|No Album|No Artist)//ig;
		$tmpFormattedString =~ s/\W//g;

		if (! $tmpFormattedString) {
			# Fallback for streams: display playlist name (if available)
			if ($client->currentPlaylist() && $prefs->client($client)->get('plugin_musicinfo_stream_fallback') && ($string = Slim::Music::Info::standardTitle($client, $client->currentPlaylist()))) {
				$log->debug("empty string being replaced by playlist name");
				$formattedString = $string;
			}
			else {
				$log->debug("MusicInfoSCR: there seems to be nothing left...\n");
				$formattedString = '';
			}
		}
	}
	
	$formattedString =~ s/NOTHING//g;

	return $formattedString;
}

sub getCustomItem() {
	my ($client, $song, $formattedString) = @_;

	for my $item (keys %{$customItems}) {
		if ($formattedString =~ /\b$item\b/) {
			my $string;
			if ((defined $customItems->{$item}->{'cache'} && not defined $displayCache{$client}->{$item}->{'timeout'})
				|| (defined $customItems->{$item}->{'cache'} && $displayCache{$client}->{$item}->{'timeout'} && $displayCache{$client}->{$item}->{'timeout'} <= time())
				|| ((not defined $customItems->{$item}->{'cache'}) && $displayCache{$client}->{$item}->{'key'} ne Slim::Music::Info::getCurrentTitle($client, $song->url) . $song->url)) {

				$string = eval { &{$customItems->{$item}->{'cb'}}($client, $song) };
				$log->debug("Problem executing callback: $@") if ($@ && $log->is_debug);
				$displayCache{$client}->{$item}->{'value'} = $string;
				
				# store some caching triggers
				if (defined $customItems->{$item}->{'cache'}) {
					$displayCache{$client}->{$item}->{'timeout'} = time() + $customItems->{$item}->{'cache'};
				}
				else {
					$displayCache{$client}->{$item}->{'key'} = Slim::Music::Info::getCurrentTitle($client, $song->url) . $song->url;
				}
			}
			else {
				$string = $displayCache{$client}->{$item}->{'value'};
			}

			$formattedString =~ s/\b$item\b/$string/g;			
		}
	}

	return $formattedString;
}

sub getPlayTime {
	my ($client, $part1, $part2, $line, $song, $nowPlaying) = @_;

	return $part1 unless ($part1 =~ /(PLAYTIME|PROGRESSBAR)/);
	
	# in playlist mode don't display playtime & co., but only SONGTIME
	if (!$nowPlaying) {
		# don't display a progress bar in playlist mode
		$part1 =~ s/PROGRESSBAR//g;
		
		# in playlist mode always display SONGTIME only, when PLAYTIME is defined 
		my $songtime = getSongTime($song);
		$part1 =~ s/\bPLAYTIME\b/$songtime/g;
	}
	
	my %parts;

	if ($part1 && ($part1 =~ /PLAYTIME/)) {
		my $withoutTag = $part1;

		# if we want the progressbar, do some width calculation
		if ($part1 =~ /PLAYTIME_PROGRESS/) {
			$withoutTag =~ s/PLAYTIME_PROGRESS//;

			if ($line == 2 && $client->measureText(' ', 1)) {
				# pad string if left part is smaller than half the display size
				if ($client->measureText($part2, $line) < ($client->displayWidth()/2)) {
					$part2 = pack('A' . int(($client->displayWidth() / 1.7) / $client->measureText(' ', 1)), ' ');
				}
				else {
					$part2 = pack('A' . int($client->measureText("$part2  X", 2) / $client->measureText(' ', 1)), ' ');
				}
			}
			else {
				$part2 .= ' ';
			}
		}

		$parts{'line'}[0] = $part2 . $withoutTag;
		$client->nowPlayingModeLines( \%parts );

		# remove the progress bar if we only want the playtime and playtime is displayed (not buffer)
		if ($part1 !~ /PLAYTIME_PROGRESS/ && $parts{'overlay'}[0] =~ /\d{1,2}\:\d{1,2}/) {
			# remove anything that's not part of a usual time display...
			$parts{'overlay'}[0] =~ s/[^\-\d\:]//g;
		}

		$part1 =~ s/PLAYTIME_PROGRESS|PLAYTIME/$parts{'overlay'}[0]/g;
	}

	# use match so we get just the progress bar even if paused/stopped
	if ($part1 and ($part1 =~ /PROGRESSBAR/)) {
		if (Slim::Player::Source::playingSongDuration($client)) {
			# below copied from nowPlayingModeLines, otherwise would have to reset the now playing mode
			my $withoutTag = $part1;
			$withoutTag =~ s/PROGRESSBAR//;
			$part2 .= ' ' . $withoutTag if ($part2);

			my $fractioncomplete = Slim::Player::Source::progress($client);
			my $barlen = $client->displayWidth() - $client->measureText($part2, $line);
			
			# in single line mode (double sized font), half the length - progressBar does not seem to be aware of this
			$barlen /= 2 if (not $client->measureText(' ', 1));
			$part2 = $client->symbols($client->progressBar($barlen, $fractioncomplete));
			$part1 =~ s/PROGRESSBAR/$part2/;
		} else {
			# don't print anything if there's no duration
			$part1 =~ s/PROGRESSBAR//;
		}
	}
	
	# remove trailing/leading blanks
	$part1 =~ s/^\s*(\S*)\s$/$1/;

	return $part1;
}

# mostly taken from Slim::Player::Player::nowPlayingModeLines
sub getBufferFullness {
	my ($client, $url) = @_;
	$url = $url->url;

	my $fractioncomplete = 0;

	$fractioncomplete = $client->usage();
	my $songtime = int($fractioncomplete * 100 + 0.5)."%";
	
	# for remote streams where we know the bitrate, 
	# show the number of seconds of audio in the buffer instead of a percentage
	if ( Slim::Music::Info::isRemoteURL($url) ) {
		my $decodeBuffer;
		
		# Display decode buffer as seconds if we know the bitrate, otherwise show KB
		my $bitrate = Slim::Music::Info::getBitrate($url);
		if ( $bitrate > 0 ) {
			$decodeBuffer = sprintf( "%.1f", $client->bufferFullness() / ( int($bitrate / 8) ) );
		}
		else {
			$decodeBuffer = sprintf( "%d KB", $client->bufferFullness() / 1024 );
		}
		
		my $outputBuffer = $client->outputBufferFullness() / (44100 * 8);
		$songtime  = ' ' . sprintf "%s / %.1f", $decodeBuffer, $outputBuffer;
		$songtime .= ' ' . $client->string('SECONDS');
	}
	
	return $songtime;
}

sub getSongTime {
	my $song = shift;

	my $songduration = Slim::Music::Info::getDuration($song);
	my $hrs = int($songduration / (60 * 60));
	my $min = int(($songduration - $hrs * 60 * 60) / 60);
	my $sec = $songduration - ($hrs * 60 * 60 + $min * 60);
	
	if ($hrs) {
		$songduration = sprintf("%d:%02d:%02d", $hrs, $min, $sec);
	} else {
		$songduration = sprintf("%02d:%02d", $min, $sec);
	}

	return $songduration;
}	

sub getKBPS {
	my $song = shift;

	my $bitrate = Slim::Music::Info::getBitrate($song);
	my $kbps = int($bitrate / 1000);

	return $kbps . string('KBPS');
}

sub buttonIcons {
	my ($client, $position, $line) = @_;
	
	return '' unless ($displayCache{$client}->{'graphical'});

	my $shuffleState = $serverPrefs->client($client)->get('shuffle');
	my $repeatState = $serverPrefs->client($client)->get('repeat');

	if (($shuffleState != $displayCache{$client}->{'shuffle'})
	 || ($repeatState != $displayCache{$client}->{'repeat'})
	 || ($line ne $displayCache{$client}->{'line'})
	 || ($position ne $displayCache{$client}->{'position'})) {

		# TOP*2 means always display the icon, while TOP* only 
		# does when some shuffle/repeat mode is set
		if ($shuffleState || ($position =~ /TOP(LEFT|RIGHT)2/)) {
			$line .= addIcon($client, chr(4 + $shuffleState));
		}

		if ($repeatState || ($position =~ /TOP(LEFT|RIGHT)2/)) {
			$line .= addIcon($client, chr(1 + $repeatState));
		}

		$displayCache{$client}->{'repeat'} = $repeatState;
		$displayCache{$client}->{'shuffle'} = $shuffleState;
		$displayCache{$client}->{'position'} = $position;
		$displayCache{$client}->{'line'} = $line;
	}

	return $line;
}

sub sleepIcons {
	my ($client, $position, $line) = @_;

	my $remaining = $client->sleepTime;

	return $line unless ($client->isa('Slim::Player::Squeezebox2') && $remaining);

	$remaining = $remaining - time();
	my @sleepChoices = (90, 60, 45, 30, 15, 10, 5);
	my $sleepState = 0;

	# find the next value for the sleep timer
	for ($sleepState = $#sleepChoices; $sleepState >= 0; $sleepState--) {

		if ( $remaining <= $sleepChoices[$sleepState]*60 ) {
			last;
		}
	}

	# TOP*2 means show animated icon
	if ($position =~ /TOP(LEFT|RIGHT)2/ && $sleepState <= $#sleepChoices) {

		$line .= addIcon($client, chr(8 + ($sleepState*2) + ($remaining%2)));
	}

	elsif ($position =~ /TOP(LEFT|RIGHT)$/ || $sleepState == $#sleepChoices) {

		$line .= addIcon($client, chr(7));
	}

	return $line;
}

sub addIcon {
	my ($client, $symbol) = @_;
	my $display = $client->display;
	
	return $display->symbols('tight') . $display->symbols('font')
		. 'MIScrIcons' . $displayCache{$client}->{'model'} . '.1' .$display->symbols('/font') 
		. chr(0) . $symbol . $display->symbols('defaultfont') . $display->symbols('/tight');	
}

1;
