# Biography::Plugin.pm by mh 2005
#
# This code is derived from code with the following copyright message:
#
# SliMP3 Server Copyright (C) 2001 Sean Adams, Slim Devices Inc.
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License,
# version 2.
#
# Changelog
# 0.1 - first release
# 0.2 - added support for proxy server (thanks to Jaimes Craig for pointing this out)
#     - added _blank target to non relative external links
#     - added configuration options for slideshow mode, web interface refresh, cache purge
#     - added regular cache purge 
#     - fixed crash when there's no player available
#     - added some more informational texts (nothing found, gathering information etc.)

package Plugins::Biography::Plugin;

use strict;

use vars qw($VERSION);
$VERSION = substr(q$Revision: 0.1 $,10);

use File::Spec;
use HTML::FormatText;

use Slim::Buttons::Common;
use Slim::Display::Animation;
use Slim::Music::Info;
use Slim::Player::Playlist;
use Slim::Player::Protocols::HTTP;
use Slim::Utils::Prefs;
use Slim::Utils::Strings qw(string);

my $baseURL    = 'http://www.allmusic.com';
my $searchURL  = "$baseURL/cg/amg.dll";
my $ignoreList = 'quartet band group the';

my $pluginFolder = 'Plugins/Biography';
my $stringsFile  = "$pluginFolder/biography-strings.txt";

my ($cacheFolder, %functions, %formattedBio, %bioLength, %line, %status);

sub initPlugin {

	$cacheFolder = initCacheFolder();

	%functions = (
		'up' => sub  {
			my $client = shift;
			$line{$client} = Slim::Buttons::Common::scroll($client, -1, $bioLength{$client}, $line{$client} || 0);
			$client->update();
		},

		'down' => sub  {
			my $client = shift;
			$line{$client} = Slim::Buttons::Common::scroll($client, 1, $bioLength{$client}, $line{$client} || 0);
			$client->update();
		},
		
		'left' => sub  {
			my $client = shift;
			Slim::Buttons::Common::popModeRight($client);
		},

		'right' => sub  {
			my $client = shift;
			Slim::Display::Animation::bumpRight($client);
		}
	);
}

sub strings {
	local $/ = undef;
	<DATA>;
}

sub getDisplayName {
	return 'PLUGIN_BIOGRAPHY';
}

sub setMode {
	my $client = shift;
	my $method = shift;

	$client->lines( \&lines );

	$status{$client} ||= '-1';

	if ($method ne 'pop') {

		if (my $url = Slim::Player::Playlist::song($client)) {

			my $artist = _artistFromUrl($url);
	
			# only read bio if artist changed
			if ($artist ne $status{$client}) {

				# Please wait...
				$status{$client} = 0;
				$client->update();
	
				my @formattedBio = ();
				my $artistHTML   = getArtistInfo($artist);
	
				if ((!$artistHTML) || ($artistHTML =~ /^<!-- NOT FOUND -->/)) {

					push @formattedBio, $client->string('PLUGIN_BIOGRAPHY_NOTHINGFOUND');

				} else {

					my $bioFormatter = HTML::FormatText->new(leftmargin => 0, rightmargin => 39);
					my $formattedBio = getArtistBio($artistHTML);

					$formattedBio =~ s/Read More\.\.\./(Read more on the web)/i;
					$formattedBio = $bioFormatter->format_string($formattedBio);
					$formattedBio =~ s/\n+/\n/g;

					@formattedBio = split(/\n/, $formattedBio);

					if (@formattedBio == 0) {
						push(@formattedBio, string('PLUGIN_BIOGRAPHY_NOTEXTFOUND'));
					} else {
						push(@formattedBio, '----------------------------------------');
					}
				}
	
				$bioLength{$client} = $#formattedBio + 1;
				$formattedBio{$client} = \@formattedBio;
				$line{$client} = 0;
			}

			$status{$client} = $artist;

		} else {

			$status{$client} = '-1';
		}

		$client->update();
	}
}

sub lines {
	my $client = shift;

	my $line1 = '';
	my $line2 = '';
	
	if ($status{$client} eq "0") {

		$line1 = $client->string('PLUGIN_BIOGRAPHY_COLLECTINGINFO');

	} elsif ($status{$client} eq "-1") {

		$line1 = $client->string('PLUGIN_BIOGRAPHY');
		$line2 = $client->string('PLUGIN_BIOGRAPHY_NOTHINGFOUND');

	} else {

		$line1 = $client->string('PLUGIN_BIOGRAPHY') . ' - ' . $status{$client};
		$line2 = $formattedBio{$client}[$line{$client}];
	}

	return ($line1, $line2);
}

sub getFunctions {
	return \%functions;
}

sub setupGroup {

	my %setupGroup = (
		PrefOrder => ['plugin_biography_slideshow', 'plugin_biography_refresh', 'plugin_biography_cache', 'plugin_biography_purge_cache'],
		GroupHead => string('PLUGIN_BIOGRAPHY'),
		GroupLine => 1,
		GroupSub => 0,
		Suppress_PrefSub => 0,
		Suppress_PrefLine => 1
	);

	my %setupPrefs = (

		'plugin_biography_slideshow' => {
			'validate' => \&Slim::Web::Setup::validateInt,
			'validateArgs' => [0,undef,1]
		},

		'plugin_biography_refresh' => {
			'validate' => \&Slim::Web::Setup::validateInt,
			'validateArgs' => [0,undef,1]
		},

		'plugin_biography_cache' => {
			'validate' => \&Slim::Web::Setup::validateInt,
			'validateArgs' => [1,undef,1],

			'onChange' => sub {

				my ($client,$changeref,$paramref,$pageref) = @_;

				Slim::Utils::Prefs::set('plugin_biography_cache', $changeref->{'plugin_biography_cache'}->{'new'});

				initCacheFolder();
			}
		}
	);

	return (\%setupGroup, \%setupPrefs);
}

sub webPages {
	my %pages = ("index\.htm" => \&handleWebIndex);
	return (\%pages, "index.html");
}

sub handleWebIndex {
	my ($client, $params) = @_;

	if (!$client) {

		$params->{'artist'}    = string('NO_PLAYER_FOUND');
		$params->{'artistbio'} = string('NO_PLAYER_DETAILS');

	} elsif (my $url = Slim::Player::Playlist::song($client)) {

		_artistFromUrl($url, $params);
	
		if (Slim::Utils::Prefs::get('plugin_biography_refresh')) {

			$params->{'refresh'} = Slim::Utils::Prefs::get('plugin_biography_refresh') * 1000;
		}
	
		my $artistHTML = getArtistInfo($params->{'artist'});

		if (!$artistHTML) {

			$params->{'artistbio'} = $client->string('PLUGIN_BIOGRAPHY_NOTHINGFOUND');

		} elsif ($artistHTML =~ /^<!-- NOT FOUND -->/) {

			$params->{'artistbio'} = $artistHTML;

		} else {

			$params->{'artistbio'}    = getArtistBio($artistHTML, $params->{'artist'});
			$params->{'artistimages'} = getArtistImages($artistHTML);
			$params->{'artistlinks'}  = getArtistLinks($artistHTML);
		}

	} else {

		$params->{'artist'} = $client->string('PLUGIN_BIOGRAPHY_NOTHINGFOUND');
	}

	return Slim::Web::HTTP::filltemplatefile('plugins/Biography/index.html', $params);
}

sub _artistFromUrl {
	my ($url, $params) = @_;

	my $ds    = Slim::Music::Info::getCurrentDataStore();
	my $track = $ds->objectForUrl($url);

	$params->{'artist'} = $track->artist();

	# We didn't get an artist - try to extract one from the title.
	unless ($params->{'artist'}) {

		$params->{'artist'} = $track->title();
		$params->{'artist'} =~ /([^\-]+?) -/;

		if ($1) {
			$params->{'artist'} = $1;
		}
	}

	return $params->{'artist'};
}

sub getArtistInfo {
	my $searchArtist = shift;

	my $cachedFile   = "$cacheFolder/$searchArtist.html";

	# Try and use the cached copy first.
	if (-e $cachedFile && -M $cachedFile < 30) {

		local $/ = undef;

		open CACHE, $cachedFile;
		my $cachedContent = <CACHE>;
		close CACHE;

		return $cachedContent;
	}

	for (split(/ /, $ignoreList)) {
		$searchArtist =~ s/\b$_\Z//sig;
	}

	chomp($searchArtist);
	
	my $request = "SQL=$searchArtist&OPT1=1&submit=Go&P=amg";

	# This makes a non-blocking request to the remote server.
	my $http = Slim::Player::Protocols::HTTP->new({
		'url'  => $searchURL,
		'post' => $request,
	}) || return 'timeout?';

	my $html = $http->content();
	$http->close();

	# direct hit?
	if (($html =~ /Overview/si) && ($html =~ /Biography/si) && ($html =~ /Discography/si)) {

		storeCachedFile($cachedFile, $html);
		return $html;

	} else {
		$html =~ /<a href="(\/cg\/amg.dll\?p=amg\&sql=\d{1,2}:\w+)">(.*?)<\/a>/si;
		
		my $artistURL = $1;
		my $artistFound = $2;
		my $ok = 1;

		# verify if we really match the search expression
		for (split(/[\W]/, $searchArtist))	{

			if ($_ && $artistFound !~ /$_/i) {
				$ok = 0;
				last;
			}
		}

		# give the first entry a try
		if ($ok || ($searchArtist =~ $artistFound)) {

			my $http = Slim::Player::Protocols::HTTP->new({'url' => $baseURL . $artistURL});
			my $html = $http->content();
			$http->close();

			storeCachedFile($cachedFile, $html);
			return $html;

		} else {
			# show list of possible matches
			my $htmlList = '';

			while ($html =~ /<a (href="\/cg\/amg.dll\?p=amg\&sql=\d{1,2}:\w+")>(.*?)<\/a>(<\/strong>|)<\/td><td.*?>(.*?)<\/td>/sig) {
				$htmlList .= "<li><a " . fixBaseURL($1) . ">$2 ($4)</a></li>\n";
			}

			return "<!-- NOT FOUND --><ul>$htmlList</ul>\n";
		}
	}
}

sub storeCachedFile {
	my $cachedFile    = shift;
	my $cachedContent = shift;
	
	open(CACHE, ">$cachedFile") || Slim::Utils::Misc::msg("Could not open $cachedFile for writing: $!\n");
	print CACHE $cachedContent;
	close(CACHE);
}

sub getArtistLinks {
	my $artistHTML = shift;

	if ($artistHTML =~ /<hr[^>]*?><table.*?(<table.*?)<\/td><\/tr><\/table><\/td><!--End Center Content-->/si) {

		my $html = $1;
		$html =~ s/<[\/]?(span|div)[^>]*?>//sig;

		return fixBaseURL($html);

	} else {
		return '';
	}
}


sub getArtistBio {
	my $artistHTML   = shift;
	my $searchArtist = shift;

	if ($artistHTML =~ /<div id="artistminibio">.*?<p.*?>(.*?)<\/p.*?>/si) {
		return fixBaseURL($1);
	}

	return string('PLUGIN_BIOGRAPHY_NOTHINGFOUND');
}

sub getArtistImages {

	my $artistHTML = shift;
	my $html = '<script type="text/javascript" language="javascript" src="' . $baseURL . '/js/pictures.js"></script>';
	
	if ($artistHTML !~ /Begin Page Photo-->(.*?)<!--End Page Photo/si) {
		return '';
	}

	$1 =~ /<table[^>]*?>(.*?)<\/table[^>]*?>/i;
	$html .= "\n\n$1\n\n";
	$html = "<table cellpadding=\"0\" cellspacing=\"0\">\n$html</table>";
	
	# ...the picture list...
	if ($artistHTML =~ /^(Pictures.*?\n*^Credits.*?)$/msi) {

		my $imgList = $1;
		   $imgList =~ /Pictures = new Array\(([^\)]*?)\);/si;

		my @images   = split /,/, $1;
		my $startImg = int rand @images;

		$imgList .= "\nvar i = $startImg;\n";

		if (Slim::Utils::Prefs::get('plugin_biography_slideshow')) {
			$imgList .= "nextPicture();\nsetInterval(\"nextPicture()\", " . 
				Slim::Utils::Prefs::get('plugin_biography_slideshow') * 1000 . ");\n";
		}

		$html .= "<script type=\"text/javascript\" language=\"javascript\">\n<!--\n$imgList\n-->\n</script>\n";
	}

	return $html;
}

sub fixBaseURL {
	my $html = shift;

	$html =~ s/href="(.*?allmovie.com.*?)"/href="$1" target="_blank"/sig;
	$html =~ s/href="\/(.*?)"/href="$baseURL\/$1" target="_blank"/sig;

	return $html;
}

sub initCacheFolder {
	my $cacheFolder = Slim::Utils::Prefs::get('cachedir');

	mkdir($cacheFolder) unless (-d $cacheFolder);

	$cacheFolder .= "/.slimserver-biography-cache";

	mkdir($cacheFolder) unless (-d $cacheFolder);
	
	my $cacheAge = Slim::Utils::Prefs::get('plugin_biography_cache') || 30;
	
	# purge the cache
	opendir(DIR, $cacheFolder) or die "can't opendir $cacheFolder: $!";
	while (defined(my $cachedFile = readdir(DIR))) {

		$cachedFile = "$cacheFolder/$cachedFile";

		if (-M $cachedFile > $cacheAge) {
			unlink $cachedFile;
		}
	}

	closedir(DIR);
	
	# set timer to purge cache once a day
	Slim::Utils::Timers::setTimer(0, Time::HiRes::time() + 60*60*24, \&initCacheFolder);
	
	return $cacheFolder;
}

1;

__DATA__
PLUGIN_BIOGRAPHY
	DE	Biographie
	EN	Biography

PLUGIN_BIOGRAPHY_RELOAD
	DE	Aktualisieren
	EN	Refresh

PLUGIN_BIOGRAPHY_NOTHINGFOUND
	DE	Konnte keine Informationen finden!
	EN	Couldn't find any information!

PLUGIN_BIOGRAPHY_NOTEXTFOUND
	DE	Konnte keinen Biographie-Text finden!
	EN	Couldn't find any biography text!

PLUGIN_BIOGRAPHY_COLLECTINGINFO
	DE	Suche Informationen...
	EN	Gathering information...

SETUP_PLUGIN_BIOGRAPHY_SLIDESHOW
	DE	Dia-Show
	EN	Slide show mode

SETUP_PLUGIN_BIOGRAPHY_SLIDESHOW_CHOOSE
	DE	Bildwechsel-Intervall in Sekunden:
	EN	Image change interval in seconds:

SETUP_PLUGIN_BIOGRAPHY_SLIDESHOW_DESC
	DE	Falls mehrere Bilder gefunden werden, so kann mit einem Mausklick auf das Bild zwischen diesen gewechselt werden. Sie knnen diesen Wechsel aber auch wie bei einer Diashow automatisieren. Geben Sie ein Zeitinterval an, in welchem die Bilder gewechselt werden sollen. \"0\" deaktiviert den automatischen Wechsel. (Vorgabe: 0)
	EN	There may be more than one picture for an interpret. You can change between them by clicking on the image or automatically by defining a change interval. Put 0 if you don't want the pictures to flick automatically or any other value for the change interval in seconds. (Default: 0)

SETUP_PLUGIN_BIOGRAPHY_REFRESH
	DE	Aktualisierungs Intervall
	EN	Refresh interval

SETUP_PLUGIN_BIOGRAPHY_REFRESH_CHOOSE
	DE	Aktualisierungs-Intervall in Sekunden:
	EN	Refresh interval in seconds:

SETUP_PLUGIN_BIOGRAPHY_REFRESH_DESC
	DE	Sie knnen die Biographie im Web-Interface automatisch aktualisieren lassen, um sie mit der aktuell gespielten Musik zu synchronisieren. Geben Sie ein Zeitinterval an, in welchem die Biographie aktualisiert werden soll. \"0\" deaktiviert die automatische Aktualisierung. (Vorgabe: 0)
	EN	You can have the biography in your web interface update automatically to stay in sync with the currently playing song. Define the refresh interval in seconds or 0 if you don't want it to be updated automatically. (Default: 0)

SETUP_PLUGIN_BIOGRAPHY_CACHE
	DE	Biographie Cache
	EN	Biography cache

SETUP_PLUGIN_BIOGRAPHY_CACHE_CHOOSE
	DE	Anzahl Tage, die die Biographien lokal gespeichert bleiben sollen:
	EN	Days to keep local copies of biographies:

SETUP_PLUGIN_BIOGRAPHY_CACHE_DESC
	DE	Die Biographie-Daten werden im Verzeichnis $cacheFolder zwischengespeichert, um den Zugriff zu beschleunigen. Definieren Sie hier, wieviele Tage die Daten lokal gespeichert werden sollen.
	EN	The artist's biographies are cached locally in $cacheFolder to speed up the display. Define for how many days you want to keep the local copies. (Default: 30)
