// some common components for the player control
SqueezeJS.UI = {
	// add some custom events we'll be using to our base class
	Component : Ext.extend(Ext.Component, {
		initComponent : function(config){
			if (typeof config == 'string')
				config = { el: config };
	
			Ext.apply(this, config);
			SqueezeJS.UI.Component.superclass.initComponent.call(this);
	
			this.el = Ext.get(this.el);
	
			// subscribe to some default events
			SqueezeJS.Controller.on({
				'playlistchange': {
					fn: this.onPlaylistChange,
					scope: this
				},
				'playerstatechange': {
					fn: this.onPlayerStateChange,
					scope: this
				}
			});
		},

		onPlaylistChange : function(){},
		onPlayerStateChange : function(){}
	}),

	// graphical button, defined in three element sprite for normal, mouseover, pressed
	Buttons : {
		Base : Ext.extend(Ext.Button, {
			power: 0,
			cmd : null,
			cmd_id : null,
			cls : '',
		
			initComponent : function(){
				this.tooltipType = this.initialConfig.tooltipType || 'title';
		
				this.template = new Ext.Template(
					'<table border="0" cellpadding="0" cellspacing="0"><tbody><tr>',
					'<td></td><td><button type="{1}" style="padding:0">{0}</button></td><td></td>',
					'</tr></tbody></table>');
		
				SqueezeJS.UI.Buttons.Base.superclass.initComponent.call(this);
			
				SqueezeJS.Controller.on({
					'playerstatechange': {
						fn: function(result){
							this.power = (result.power == null) || result.power; 
		
							// update custom handler for stations overwriting default behavior
							if (this.cmd_id && result.playlist_loop && result.playlist_loop[0] 
								&& result.playlist_loop[0].buttons && result.playlist_loop[0].buttons[this.cmd_id]) {
					
								var btn = result.playlist_loop[0].buttons[this.cmd_id];
					
								if (btn.cls)
									this.setClass(btn.cls);
								else if (btn.icon)
									this.setIcon(btn.icon);
					
								if (btn.tooltip)
									this.setTooltip(btn.tooltip);
					
								if (btn.command)
									this.cmd = btn.command;
							}
		
							this.onPlayerStateChange(result);
						},
						scope: this
					}
				});
		
				this.on({
					'render': {
						fn: function() {
							if (this.minWidth) {
								var btnEl = this.el.child("button:first");
								btnEl.setWidth(this.minWidth);
							}
						},
						scope: this
					}
				});
			},
		
			onPlayerStateChange : function(result){},
		
			setTooltip: function(tooltip){
				if (this.tooltip == tooltip)
					return;
		
				this.tooltip = tooltip;
				
				var btnEl = this.el.child("button:first");
		
				if(typeof this.tooltip == 'object'){
					Ext.QuickTips.tips(Ext.apply({
						target: btnEl.id
					}, this.tooltip));
				} 
				else {
					btnEl.dom[this.tooltipType] = this.tooltip;
				}
			},
		
			setClass: function(newClass) {
				this.el.removeClass(this.cls);
				this.cls = newClass
				this.el.addClass(this.cls);
			},
		
		
			setIcon: function(newIcon) {
				var btnEl = this.el.child("button:first");
				if (btnEl)
					btnEl.setStyle('background-image', newIcon ? 'url(' + webroot + newIcon + ')' : '');
			}
		})
	}
};

SqueezeJS.UI.ScrollPanel = {
	offset : 0,
	el : null,

	init : function() {
		var el;

		if (el = Ext.get('infoTab'))
			this.offset += el.getHeight();

		if (el = Ext.get('pageFooterInfo'))
			this.offset += el.getHeight();

		if ((el = Ext.get('browsedbList')) ||
			((el = Ext.get('content')) && el.hasClass('scrollingPanel'))) {

			this.el = el;
			this.offset += this.el.getTop();

			Ext.EventManager.onWindowResize(this.onResize, this);
			this.onResize();
		}
	},

	onResize : function(){
		this.el.setHeight( Ext.fly(document.body).getViewSize().height - this.offset );
	}
};

SqueezeJS.UI.Highlight = {
	highlightedEl : null,
	unHighlightTimer : null,
	isDragging : false,

	init : function() {
		// make sure all selectable list items have a unique ID
		var items = Ext.DomQuery.select('.selectorMarker');
		for(var i = 0; i < items.length; i++) {
			Ext.id(Ext.get(items[i]));
		}

		if (!this.unHighlightTimer)
			this.unHighlightTimer = new Ext.util.DelayedTask(this.unHighlight, this);

		// don't remove the highlight automatically while we're editing a search term or similar
		Ext.select('.browsedbControls input[type="text"]').on({
			focus: this.unHighlightTimer.cancel,
			click: this.unHighlightTimer.cancel
		});
	},

	highlight : function(target, onClickCB){
		// don't highlight while dragging elements around
		if (this.isDragging)
			return;

		// remove highlighter after x seconds of inactivity
		if (this.unHighlightTimer) {
			this.unHighlightTimer.cancel();
		}

		// return if the target is a child of the main selector
		var el = Ext.get(target.id); 
		if (el == this.highlightedEl)
			return;

		// always highlight the main selector, not its children
		if (el != null) {
			this.unHighlight();
			this.highlightedEl = el;

			el.replaceClass('selectorMarker', 'mouseOver');

			this.highlightedEl.onClickCB = onClickCB || this.onSelectorClicked;
			el.on('click', this.highlightedEl.onClickCB);
		}
	},

	onMouseOut : function() {
		// remove the highlight in a bit
		if (this.unHighlightTimer)
			this.unHighlightTimer.delay(500);
	},
	
	unHighlight : function(){
		// remove highlighting from the other DIVs
		if (this.highlightedEl) {
			this.highlightedEl.replaceClass('mouseOver', 'selectorMarker');
			this.highlightedEl.un('click', this.highlightedEl.onClickCB);
			this.highlightedEl = null;
		}
	},

	onSelectorClicked : function(ev, target){
		target = Ext.get(target);
		if (target.hasClass('browseItemDetail') || target.hasClass('playlistSongDetail'))
			target = Ext.get(target.findParentNode('div'));

		var el = target.child('a.browseItemLink');
		if (el && el.dom.href) {
			if (el.dom.target) {
				try { parent.frames[el.dom.target].location.href = el.dom.href; }
				catch(e) { location.href = el.dom.href; }
			}
			else {
				location.href = el.dom.href;
			}
		}
	}
}

// common button and label components, automatically updated on player events
SqueezeJS.UI.Buttons.Play = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	isPlaying: false,

	initComponent : function(){
		this.cls = this.cls || 'btn-play'; 
		this.tooltip = this.tooltip || SqueezeJS.string('play');
		SqueezeJS.UI.Buttons.Play.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.isPlaying) {
			this.updateState(false);
			SqueezeJS.Controller.playerControl(['pause']);
		}
		else {
			this.updateState(true);
			SqueezeJS.Controller.playerControl(['play']);
		}
	},

	onPlayerStateChange: function(result){
		var newState = (result.mode == 'play');

		if (this.isPlaying != newState) {
			this.updateState(newState);
		}
	},

	updateState: function(isPlaying){
		var playEl = Ext.get(Ext.DomQuery.selectNode('table:first', Ext.get('ctrlTogglePlay').dom));

		playEl.removeClass(['btn-play', 'btn-pause']);
		playEl.addClass(isPlaying ? 'btn-pause' : 'btn-play');

		this.setTooltip(isPlaying ? SqueezeJS.string('pause') : SqueezeJS.string('play'));
		this.isPlaying = isPlaying;
	}
});

SqueezeJS.UI.Buttons.Rew = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-previous'; 
		this.tooltip = this.tooltip || SqueezeJS.string('previous');
		SqueezeJS.UI.Buttons.Rew.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power)
			SqueezeJS.Controller.playerControl(['button', 'jump_rew']);
	},

	onPlayerStateChange: function(result){
		if (result.playlist_loop && result.playlist_loop[0] && result.playlist_loop[0].buttons) {
			try { this.setDisabled(!result.playlist_loop[0].buttons.rew) }
			catch(e){}
		}
		else if (this.disabled)
			this.enable();
	}
});

SqueezeJS.UI.Buttons.Fwd = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-next';
		this.tooltip = this.tooltip || SqueezeJS.string('next');
		SqueezeJS.UI.Buttons.Fwd.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power)
			SqueezeJS.Controller.playerControl(['button', 'jump_fwd']);
	}
});

SqueezeJS.UI.Buttons.Repeat = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	cmd_id: 'repeat',
	state: 0,

	initComponent : function(){
		this.cls = this.cls || 'btn-repeat-0';
		this.tooltip = this.tooltip || SqueezeJS.string('repeat');
		SqueezeJS.UI.Buttons.Repeat.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power) {
			if (this.cmd)
				SqueezeJS.Controller.playerControl(this.cmd);
			else
				SqueezeJS.Controller.playerControl(['playlist', 'repeat', (this.state + 1) % 3]);
		} 
	},

	onPlayerStateChange: function(result){
		if (this.cmd) {}
		else if (this.state == -1 || (result['playlist repeat'] != null && this.state != result['playlist repeat']))
			this.updateState(result['playlist repeat']);

	},

	updateState: function(newState){
		this.state = newState || 0;
		this.setIcon('');
		this.setTooltip(SqueezeJS.string('repeat' + ' - ' + SqueezeJS.string('repeat' + this.state)));
		this.setClass('btn-repeat-' + this.state);
	}
});

SqueezeJS.UI.Buttons.Shuffle = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	cmd_id: 'shuffle',
	state: 0,

	initComponent : function(){
		this.cls = this.cls || 'btn-shuffle-0';
		this.tooltip = this.tooltip || SqueezeJS.string('shuffle');
		SqueezeJS.UI.Buttons.Shuffle.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power) {
			if (this.cmd)
				SqueezeJS.Controller.playerControl(this.cmd);
			else
				SqueezeJS.Controller.playerControl(['playlist', 'shuffle', (this.state + 1) % 3]);
		} 
	},

	onPlayerStateChange: function(result){
		if (this.cmd) {}
		else if (this.state == -1 || (result['playlist shuffle'] != null && this.state != result['playlist shuffle']))
			this.updateState(result['playlist shuffle']);

	},

	updateState: function(newState){
		this.state = newState || 0;
		this.setIcon('');
		this.setTooltip(SqueezeJS.string('shuffle' + ' - ' + SqueezeJS.string('shuffle' + this.state)));
		this.setClass('btn-shuffle-' + this.state);
	}
});

SqueezeJS.UI.Buttons.Power = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-power';
		this.tooltip = this.tooltip || SqueezeJS.string('power');
		SqueezeJS.UI.Buttons.Power.superclass.initComponent.call(this);
	},

	handler: function(){
		var newState = (this.power ? '0' : '1');
		this.power = !this.power;
		this.onPlayerStateChange();
		SqueezeJS.Controller.playerControl(['power', newState]);
	},

	onPlayerStateChange: function(result){
		this.setTooltip(SqueezeJS.Strings['power'] + SqueezeJS.Strings['colon'] + ' ' + SqueezeJS.Strings[this.power ? 'on' : 'off']);

		if (this.power)
			this.el.removeClass('btn-power-off');
		else
			this.el.addClass('btn-power-off');
	}
});

SqueezeJS.UI.Buttons.PlayerDropdown = Ext.extend(Ext.SplitButton, {
	playerList : null,

	initComponent : function(){
		Ext.apply(this, {
			menu: new Ext.menu.Menu({shadow: Ext.isGecko && Ext.isMac ? true : 'sides'}),
			tooltip: SqueezeJS.Strings['choose_player'],
			arrowTooltip: SqueezeJS.Strings['choose_player'],
			tooltipType: 'title'
		})
		SqueezeJS.UI.Buttons.PlayerDropdown.superclass.initComponent.call(this);

		SqueezeJS.Controller.on({
			'playerlistchange': {
				fn: this.onPlayerlistChange,
				scope: this
			}
		});
	},

	handler : function(ev){
		if(this.menu && !this.menu.isVisible()){
			this.menu.show(this.el, this.menuAlign);
		}
		this.fireEvent('arrowclick', this, ev);
	},

	onPlayerlistChange : function(response){
		this.menu.removeAll();
		this.menu.add(
			'<span class="menu-title">' + SqueezeJS.Strings['choose_player'] + '</span>'
		);

		// let's set the current player to the first player in the list
		if (response['player count'] > 0 || response['sn player count'] > 0) {
			var playerInList = false;
			var el;

			this.playerList = new Ext.util.MixedCollection();

			this._addPlayerlistMenu(response);
			this._addSNPlayerlistMenu(response);

			// add the sync option menu item
			this.menu.add(
				'-',
				new Ext.menu.Item({
					text: SqueezeJS.Strings['synchronize'] + '...',
					// query the currently synced players and show the dialog
					handler: function(){
						SqueezeJS.Controller.playerRequest({
							params: ['sync', '?'],
							success: this.showSyncDialog,
							failure: this.showSyncDialog
						});	
					},
					disabled: (this.playerList.getCount() < 2) 
				})
			);

//			if (!playerInList)
//				this.selectPlayer();

//			else if (playerInList && !playerid && Utils.getCookie('SqueezeCenter-player'))
//				this.selectPlayer(playerInList);
		}

		else {
			this.menu.add(
				new Ext.menu.Item({
					text: SqueezeJS.Strings['no_player'] + '..',
					handler: function(){
						var dlg = new Ext.BasicDialog('', {
							autoCreate: true,
							title: SqueezeJS.Strings['no_player'],
							modal: true,
							closable: false,
							collapsible: false,
							width: 500,
							height: 250,
							resizeHandles: 'se'
						});
						dlg.addButton(SqueezeJS.Strings['close'], dlg.destroy, dlg);
						dlg.addKeyListener(27, dlg.destroy, dlg);
						dlg.body.update(SqueezeJS.Strings['no_player_details']);
						dlg.show();
					}
				})
			);

//			var el = Ext.get('playList');
//			if (el && (el = el.child('div.noPlayerPanel')))
//				el.setDisplayed(true);

			this.selectPlayer();
		}

	},

	_addPlayerlistMenu : function(response){
		if (response.players_loop) {
			for (var x=0; x < response.players_loop.length; x++) {
				var playerInfo = response.players_loop[x];

				// mark the current player as selected
				if (playerInfo.playerid == SqueezeJS.Controller.getPlayer()) {
					playerInList = {
						text: playerInfo.name,
						value: playerInfo.playerid,
						canpoweroff: playerInfo.canpoweroff
					};
					this.setText(playerInfo.name);
				}

				// add the players to the list to be displayed in the synch dialog
				this.playerList.add(playerInfo.playerid, {
					name: playerInfo.name,
					isplayer: playerInfo.isplayer
				});

				this.menu.add(
					new Ext.menu.CheckItem({
						text: playerInfo.name,
						value: playerInfo.playerid,
						canpoweroff: playerInfo.canpoweroff,
						cls: 'playerList',
						group: 'playerList',
						checked: playerInfo.playerid == playerid,
						handler: this.selectPlayer
					})
				);
			}
		}
	},

	_addSNPlayerlistMenu : function(response){
		// add a list of players connected to SQN, if available
		if (response.sn_players_loop) {
			var first = true;
							
			for (var x=0; x < response.sn_players_loop.length; x++) {
				var playerInfo = response.sn_players_loop[x];

				// don't display players which are already connected to SC
				// this is to prevent double entries right after a player has switched
				if (! this.playerList.get(playerInfo.playerid)) {
					if (first) {
						this.menu.add(
							'-',
							'<span class="menu-title">' + SqueezeJS.Strings['squeezenetwork'] + '</span>'
						);
						first = false;
					}

					this.menu.add(
						new Ext.menu.Item({
							text: playerInfo.name,
							value: playerInfo.id,
							playerid: playerInfo.playerid,
							cls: 'playerList',
							handler: this._confirmSwitchSQNPlayer
						})
					);
				}
			}
		}
	},

	selectPlayer: function(ev){
		var el;

		// currently selected player is not available. don't change anything but the button's label
		if (!ev)
			ev = {
				text: '',
				value: '',
				canpoweroff: false
			}

		this.setText(ev.text);
		SqueezeJS.Controller.setPlayer(playerid);
	},

	_confirmSwitchSQNPlayer: function(ev){
		// don't display confirmation dialog in Opera,
		// as it used to reload the full page without waiting for the answer
		if (Ext.isOpera) {
			PlayerChooser.switchSQNPlayer(ev);
		}
		else {
			Ext.MessageBox.confirm(
				strings['squeezenetwork'],
				strings['sqn_want_switch'],
				function(btn){
					if (btn == 'yes') {
						this.switchSQNPlayer(ev);
					}
				},
				this
			);
		}
	},

	_switchSQNPlayer: function(ev){
		SqueezeJS.Controller.request({ params: ['', ['squeezenetwork', 'disconnect', ev.value ]] });

		// switch player in a few seconds, to give the player time to connect
		var update = new Ext.util.DelayedTask(function(ev){
			SqueezeJS.Controller.updateAll();
			this.selectPlayer({
				text: ev.text,
				value: ev.playerid,
				canpoweroff: true
			});
		}, this, new Array(ev));
		update.delay(3000); 
	},

		showSyncDialog: function(response){
			var responseText = Ext.util.JSON.decode(response.responseText);

			var syncedPlayers = new Array();
			if (responseText.result && responseText.result._sync) {
				syncedPlayers = responseText.result._sync;
			}

			// make sure any previous syncgroup form is deleted; seems not to happen in on dlg.destroy() in some browsers
			var playerSelection = Ext.get('syncgroup');
			if (playerSelection)
				playerSelection.remove();

			playerSelection = '<form name="syncgroup" id="syncgroup">';
			var tpl = new Ext.Template('<input type="checkbox" id="{id}" value="{id}" {checked} {disabled}>&nbsp;{name}<br>');
			tpl.compile();

			// create checkboxes for other players and preselect if synced
			playerList.eachKey(function(id, data){
				if (id && data.name && id != playerid)
					playerSelection += tpl.apply({
						name: data.name,
						id: id,
						checked: parseInt(syncedPlayers.indexOf(id)) >= 0 ? 'checked' : '',
						disabled: data.isplayer ? '' : 'disabled'
					});
			});
			playerSelection += '</form>';

			var dlg = new Ext.BasicDialog('', {
				autoCreate: true,
				title: strings['synchronize'],
				modal: true,
				closable: false,
				collapsible: false,
				width: 500,
				height: 150 + playerList.getCount() * 13,
				resizeHandles: 'se'
			});

			dlg.addButton(strings['synchronize'], function(){ PlayerChooser.sync(syncedPlayers, dlg) }, dlg);
			dlg.addButton(strings['cancel'], dlg.destroy, dlg);
			dlg.addKeyListener(27, dlg.destroy, dlg);

			dlg.body.update(playerSelection);
			dlg.show();
		},

		sync: function(syncedPlayers, dlg){
			var players = Ext.query('input', Ext.get('syncgroup').dom);

			for(var i = 0; i < players.length; i++) {
				// sync if not synced yet
				if (players[i].checked && syncedPlayers.indexOf(parseInt(players[i].id)) < 0)
					Utils.processCommand({ params: [ playerid, [ 'sync', players[i].id ] ] });

				// unsync if no longer checked
				else if (syncedPlayers.indexOf(parseInt(players[i].id)) >= 0 & !players[i].checked)
					Utils.processCommand({ params: [ players[i].id, [ 'sync', '-' ] ] });
			}

			dlg.destroy();
		}
});

SqueezeJS.UI.Buttons.VolumeDown = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-volume-decrease';
		this.tooltip = this.tooltip || SqueezeJS.string('volumedown');
		SqueezeJS.UI.Buttons.VolumeUp.superclass.initComponent.call(this);
	},

	handler : function(){
		if (this.power)
			SqueezeJS.Controller.setVolume(1, '-');
	}
});

SqueezeJS.UI.Buttons.VolumeUp = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-volume-increase';
		this.tooltip = this.tooltip || SqueezeJS.string('volumeup');
		SqueezeJS.UI.Buttons.VolumeUp.superclass.initComponent.call(this);
	},

	handler : function(){
		if (this.power)
			SqueezeJS.Controller.setVolume(1, '+');
	}
});

SqueezeJS.UI.VolumeBar = Ext.extend(SqueezeJS.UI.Component, {
	power: null,
	volume : 0,

	initComponent : function(){
		SqueezeJS.UI.VolumeBar.superclass.initComponent.call(this);
	
		if (this.el && (this.el = Ext.get(this.el))) {
			var el;
			if (el = this.el.child('img:first'))
				el.on('click', this.onClick, this);
		}		
	},

	onClick: function(ev, target) {
		if (!this.power)
			return;

		var el = Ext.get(target);
		if (el) {
			var margin = 9;

			var maxWidth = el.getWidth() - 2*margin;
			var myStep = maxWidth/11;

			var myX = ev.xy[0] - el.getX() - margin - (Ext.isGecko * 5) - (Ext.isSafari * 5);
			myX = Math.max(myX, 1);
			myX = Math.min(myX, maxWidth);

			var volVal = Math.ceil(myX / myStep) - 1;

			this.updateState(volVal*10);
			SqueezeJS.Controller.setVolume(volVal);
		}
	},

	// update volume bar
	onPlayerStateChange: function(result){
		if (result['mixer volume'] != null)
			this.updateState(parseInt(result['mixer volume']));

		this.power = result.power;
	},

	updateState: function(newVolume){
		if (newVolume != this.volume) {
			var volEl;
			var volVal = Math.ceil(newVolume / 9.9); 

			if (newVolume <= 0)
				volVal = 0;
			else if (newVolume >= 100)
				volVal = 11;

			this.el.removeClass([ 'ctrlVolume0', 'ctrlVolume1', 'ctrlVolume2', 'ctrlVolume3', 'ctrlVolume4', 'ctrlVolume5', 'ctrlVolume6', 'ctrlVolume7', 'ctrlVolume8', 'ctrlVolume9', 'ctrlVolume10' ]);
			this.el.addClass('ctrlVolume' + String(Math.max(volVal-1, 0)));
	
			if (volEl = this.el.child('img:first'))
				volEl.dom.title = SqueezeJS.string('volume') + ' ' + parseInt(newVolume);

			this.volume = newVolume;
		}
	}
});


SqueezeJS.UI.Title = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.title(result, this.noLink));
	}
});

SqueezeJS.UI.CompoundTitle = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		var title = SqueezeJS.SonginfoParser.title(result, this.noLink, true);
		var contributors = SqueezeJS.SonginfoParser.contributors(result, this.noLink);
		var album = SqueezeJS.SonginfoParser.album(result, this.noLink);

		this.el.update(title
			+ (contributors ? '&nbsp;' + SqueezeJS.string('by') + '&nbsp;' + contributors : '')
			+ (album ? '&nbsp;' + SqueezeJS.string('from') + '&nbsp;' + album : '')
		);
	}
});

SqueezeJS.UI.Album = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		var year = SqueezeJS.SonginfoParser.year(result, this.noLink);
		this.el.update(SqueezeJS.SonginfoParser.album(result, this.noLink)
			+ (year ? '&nbsp;(' + year + ')' : ''));
	}
});

SqueezeJS.UI.Contributors = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.contributors(result, this.noLink));
	}
});


SqueezeJS.UI.CurrentIndex = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(parseInt(result.playlist_cur_index) + 1);
	}
});

SqueezeJS.UI.SongCount = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(parseInt(result.playlist_tracks) || 0);
	}
});

SqueezeJS.UI.Bitrate = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.bitrate(result, this.noLink));
	}
});

SqueezeJS.UI.Playtime = Ext.extend(SqueezeJS.UI.Component, {
	initComponent : function(config){
		if (typeof config == 'string')
			config = { el: config };

		Ext.apply(this, config);
		SqueezeJS.UI.Playtime.superclass.initComponent.call(this);
	
		SqueezeJS.Controller.on({
			'playtimeupdate': {
				fn: this.onPlaytimeUpdate,
				scope: this
			}
		});
	},

	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.current));
	}
});

SqueezeJS.UI.PlaytimeRemaining = Ext.extend(SqueezeJS.UI.Playtime, {
	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.remaining));
	}
});

SqueezeJS.UI.CompoundPlaytime = Ext.extend(SqueezeJS.UI.Playtime, {
	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.current) + '&nbsp;/&nbsp;' + SqueezeJS.Utils.formatTime(playtime.remaining));
	}
});

SqueezeJS.UI.PlaytimeProgress = Ext.extend(SqueezeJS.UI.Playtime, {
	initComponent : function(config){
		SqueezeJS.UI.PlaytimeProgress.superclass.initComponent.call(this);

		var el = Ext.get(this.applyTo);
		el.update( '<img src="/html/images/spacer.gif" class="progressLeft"/><img src="/html/images/spacer.gif" class="progressFillLeft"/>'
			+ '<img src="/html/images/spacer.gif" class="progressIndicator"/><img src="html/images/spacer.gif" class="progressFillRight"/>'
			+ '<img src="/html/images/spacer.gif" class="progressRight"/>' );	

		// store the DOM elements to reduce flicker
		this.remaining = Ext.get(Ext.DomQuery.selectNode('.progressFillRight', el.dom));
		this.playtime = Ext.get(Ext.DomQuery.selectNode('.progressFillLeft', el.dom));
		
		// calculate width of elements which won't be scaled
		this.fixedWidth = el.child('img.progressLeft').getWidth();
		this.fixedWidth += el.child('img.progressRight').getWidth();
		this.fixedWidth += el.child('img.progressIndicator').getWidth();

		Ext.get(this.applyTo).on('click', this.onClick);
	},

	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime) {
			var left;
			var max = this.el.getWidth() - this.fixedWidth; // total of left/right/indicator width

			// if we don't know the total play time, just put the indicator in the middle
			if (!playtime.duration)
				left = 0;

			// calculate left/right percentage
			else
				left = Math.max(
						Math.min(
							Math.floor(playtime.current / playtime.duration * max)
						, max)
					, 1);

			this.remaining.setWidth(max - left);
			this.playtime.setWidth(left);
		}
	},

	onClick : function(ev) {
		if (!SqueezeJS.Controller.playerStatus.duration)
			return;
 
		var pos = Math.max(ev.xy[0] - this.getX(), 0);
		pos = pos / Math.max(this.getWidth(), pos)
		SqueezeJS.Controller.playerControl(['time', pos * SqueezeJS.Controller.playerStatus.duration]);
	}
});

SqueezeJS.UI.Coverart = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.coverart(result, this.noLink, this.size));
	}
});

SqueezeJS.UI.CoverartPopup = Ext.extend(Ext.ToolTip, {
	initComponent : function(){
		if (this.songInfo)
			this.title = '&nbsp;';
 
		SqueezeJS.UI.CoverartPopup.superclass.initComponent.call(this);

		SqueezeJS.Controller.on({
			'playerstatechange': {
				fn: this.onPlayerStateChange,
				scope: this
			}
		});
	},

	onPlayerStateChange : function(result){
		if (this.songInfo) {
			var title = SqueezeJS.SonginfoParser.title(result, true, true);
			var contributors = SqueezeJS.SonginfoParser.contributors(result, true);
			var album = SqueezeJS.SonginfoParser.album(result, true);
	
			this.setTitle(title
				+ (contributors ? '&nbsp;/ ' + contributors : '')
				+ (album ? '&nbsp;/ ' + album : ''));
		}

		var el = this.body;
		if (el) {
			if (el = el.child('img:first', true))
				el.src = SqueezeJS.SonginfoParser.coverartUrl(result);
		}
		else {
			this.html = SqueezeJS.SonginfoParser.coverart(result, true);
		}
	}
});


SqueezeJS.UI.Playlist = Ext.extend(SqueezeJS.UI.Component, {
	initComponent : function(){
		SqueezeJS.UI.Playlist.superclass.initComponent.call(this);
		Ext.EventManager.onWindowResize(this.onResize, this);
		this.container = Ext.get(this.renderTo);
		this.onResize();
	},

	load : function(url, showIndicator){
		if (this.getPlEl())
			// unregister event handlers
			Ext.dd.ScrollManager.unregister(this.playlistEl);

		// try to reload previous page if no URL is defined
		var um = this.container.getUpdateManager();

		if (!url)
			url = um.defaultUrl;

		if (showIndicator)
			this.container.getUpdateManager().showLoadIndicator = true;

		this.container.load(
			{
				url: url || webroot + 'playlist.html?ajaxRequest=1&player=' + SqueezeJS.getPlayer(),
				method: 'GET',
				disableCaching: true
			},
			{},
			this._onUpdated.createDelegate(this)
		);

		um.showLoadIndicator = false;
	},

	getPlEl : function(){
		return Ext.get(this.playlistEl);
	},

	_onUpdated : function(o){
		this.onResize();

		// shortcut if there's no player
		if (!this.getPlEl())
			return;

		this.highlightCurrent();
	},

	onPlaylistChange : function(result) {
		this.load();
	},

	onResize : function(){
		var el = this.container.parent().parent();
		var plEl = this.getPlEl(); 
		plEl.setHeight(el.getHeight() + el.getTop() - plEl.getTop());
	},

	highlightCurrent : function(){
		var el;
		if (el = this.getPlEl()) {
			var plPos = el.getScroll();
			var plView = el.getViewSize();
			var el = Ext.DomQuery.selectNode(this.currentSelector);

			if (el) {
				el = Ext.get(el);
				if (el.getTop() > plPos.top + plView.height
					|| el.getBottom() < plPos.top)
						el.scrollIntoView(this.playlistEl);
			}
		}
	}
});