// some common components for the player control
SqueezeJS.UI = {};

// add some custom events we'll be using to our base class
SqueezeJS.UI.Component = Ext.extend(Ext.Component, {
	initComponent : function(config){
		if (typeof config == 'string')
			config = { el: config };

		Ext.apply(this, config);
		SqueezeJS.UI.Component.superclass.initComponent.call(this);

		this.el = Ext.get(this.el);

		// subscribe to some default events
		SqueezeJS.Controller.on({
			'playerlistchange': {
				fn: this.onPlayerListChange,
				scope: this
			},
			'playlistchange': {
				fn: this.onPlaylistChange,
				scope: this
			},
			'playerstatechange': {
				fn: this.onPlayerStateChange,
				scope: this
			}
		});
	},

	onPlayerListChange : function(){},
	onPlaylistChange : function(){},
	onPlayerStateChange : function(){}
});


SqueezeJS.UI.Buttons = {};

// graphical button, defined in three element sprite for normal, mouseover, pressed
SqueezeJS.UI.Buttons.Base = Ext.extend(Ext.Button, {
	power: 0,
	cmd : null,
	cmd_id : null,
	cls : '',

	initComponent : function(){
		this.tooltipType = this.initialConfig.tooltipType || 'title';

		this.template = new Ext.Template(
			'<table border="0" cellpadding="0" cellspacing="0"><tbody><tr>',
			'<td></td><td><button type="{1}" style="padding:0">{0}</button></td><td></td>',
			'</tr></tbody></table>');

		SqueezeJS.UI.Buttons.Base.superclass.initComponent.call(this);
	
		SqueezeJS.Controller.on({
			'playerstatechange': {
				fn: function(result){
					this.power = (result.power == null) || result.power; 

					// update custom handler for stations overwriting default behavior
					if (this.cmd_id && result.playlist_loop && result.playlist_loop[0] 
						&& result.playlist_loop[0].buttons && result.playlist_loop[0].buttons[this.cmd_id]) {
			
						var btn = result.playlist_loop[0].buttons[this.cmd_id];
			
						if (btn.cls)
							this.setClass(btn.cls);
						else if (btn.icon)
							this.setIcon(btn.icon);
			
						if (btn.tooltip)
							this.setTooltip(btn.tooltip);
			
						if (btn.command)
							this.cmd = btn.command;
					}

					this.onPlayerStateChange(result);
				},
				scope: this
			}
		});

		this.on({
			'render': {
				fn: function() {
					if (this.minWidth) {
						var btnEl = this.el.child("button:first");
						btnEl.setWidth(this.minWidth);
					}
				},
				scope: this
			}
		});
	},

	onPlayerStateChange : function(result){},

	setTooltip: function(tooltip){
		if (this.tooltip == tooltip)
			return;

		this.tooltip = tooltip;
		
		var btnEl = this.el.child("button:first");

		if(typeof this.tooltip == 'object'){
			Ext.QuickTips.tips(Ext.apply({
				target: btnEl.id
			}, this.tooltip));
		} 
		else {
			btnEl.dom[this.tooltipType] = this.tooltip;
		}
	},

	setClass: function(newClass) {
		this.el.removeClass(this.cls);
		this.cls = newClass
		this.el.addClass(this.cls);
	},


	setIcon: function(newIcon) {
		var btnEl = this.el.child("button:first");
		if (btnEl)
			btnEl.setStyle('background-image', newIcon ? 'url(' + webroot + newIcon + ')' : '');
	}
});

SqueezeJS.UI.Buttons.Play = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	isPlaying: false,

	initComponent : function(){
		this.cls = this.cls || 'btn-play'; 
		this.tooltip = this.tooltip || SqueezeJS.string('play');
		SqueezeJS.UI.Buttons.Play.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.isPlaying) {
			this.updateState(false);
			SqueezeJS.Controller.playerControl(['pause']);
		}
		else {
			this.updateState(true);
			SqueezeJS.Controller.playerControl(['play']);
		}
	},

	onPlayerStateChange: function(result){
		var newState = (result.mode == 'play');

		if (this.isPlaying != newState) {
			this.updateState(newState);
		}
	},

	updateState: function(isPlaying){
		var playEl = Ext.get(Ext.DomQuery.selectNode('table:first', Ext.get('ctrlTogglePlay').dom));

		playEl.removeClass(['btn-play', 'btn-pause']);
		playEl.addClass(isPlaying ? 'btn-pause' : 'btn-play');

		this.setTooltip(isPlaying ? SqueezeJS.string('pause') : SqueezeJS.string('play'));
		this.isPlaying = isPlaying;
	}
});

SqueezeJS.UI.Buttons.Rew = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-previous'; 
		this.tooltip = this.tooltip || SqueezeJS.string('previous');
		SqueezeJS.UI.Buttons.Rew.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power)
			SqueezeJS.Controller.playerControl(['button', 'jump_rew']);
	},

	onPlayerStateChange: function(result){
		if (result.playlist_loop && result.playlist_loop[0] && result.playlist_loop[0].buttons) {
			try { this.setDisabled(!result.playlist_loop[0].buttons.rew) }
			catch(e){}
		}
		else if (this.disabled)
			this.enable();
	}
});

SqueezeJS.UI.Buttons.Fwd = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-next';
		this.tooltip = this.tooltip || SqueezeJS.string('next');
		SqueezeJS.UI.Buttons.Fwd.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power)
			SqueezeJS.Controller.playerControl(['button', 'jump_fwd']);
	}
});

SqueezeJS.UI.Buttons.Repeat = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	cmd_id: 'repeat',
	state: 0,

	initComponent : function(){
		this.cls = this.cls || 'btn-repeat-0';
		this.tooltip = this.tooltip || SqueezeJS.string('repeat');
		SqueezeJS.UI.Buttons.Repeat.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power) {
			if (this.cmd)
				SqueezeJS.Controller.playerControl(this.cmd);
			else
				SqueezeJS.Controller.playerControl(['playlist', 'repeat', (this.state + 1) % 3]);
		} 
	},

	onPlayerStateChange: function(result){
		if (this.cmd) {}
		else if (this.state == -1 || (result['playlist repeat'] != null && this.state != result['playlist repeat']))
			this.updateState(result['playlist repeat']);

	},

	updateState: function(newState){
		this.state = newState || 0;
		this.setIcon('');
		this.setTooltip(SqueezeJS.string('repeat' + ' - ' + SqueezeJS.string('repeat' + this.state)));
		this.setClass('btn-repeat-' + this.state);
	}
});

SqueezeJS.UI.Buttons.Shuffle = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	cmd_id: 'shuffle',
	state: 0,

	initComponent : function(){
		this.cls = this.cls || 'btn-shuffle-0';
		this.tooltip = this.tooltip || SqueezeJS.string('shuffle');
		SqueezeJS.UI.Buttons.Shuffle.superclass.initComponent.call(this);
	},

	handler: function(){
		if (this.power) {
			if (this.cmd)
				SqueezeJS.Controller.playerControl(this.cmd);
			else
				SqueezeJS.Controller.playerControl(['playlist', 'shuffle', (this.state + 1) % 3]);
		} 
	},

	onPlayerStateChange: function(result){
		if (this.cmd) {}
		else if (this.state == -1 || (result['playlist shuffle'] != null && this.state != result['playlist shuffle']))
			this.updateState(result['playlist shuffle']);

	},

	updateState: function(newState){
		this.state = newState || 0;
		this.setIcon('');
		this.setTooltip(SqueezeJS.string('shuffle' + ' - ' + SqueezeJS.string('shuffle' + this.state)));
		this.setClass('btn-shuffle-' + this.state);
	}
});

SqueezeJS.UI.Buttons.Power = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-power';
		this.tooltip = this.tooltip || SqueezeJS.string('power');
		SqueezeJS.UI.Buttons.Power.superclass.initComponent.call(this);
	},

	handler: function(){
		var newState = (this.power ? '0' : '1');
		this.power = !this.power;
		this.onPlayerStateChange();
		SqueezeJS.Controller.playerControl(['power', newState]);
	},

	onPlayerStateChange: function(result){
		this.setTooltip(strings['power'] + strings['colon'] + ' ' + strings[this.power ? 'on' : 'off']);

		if (this.power)
			this.el.removeClass('btn-power-off');
		else
			this.el.addClass('btn-power-off');
	}
});

SqueezeJS.UI.Buttons.VolumeDown = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-volume-decrease';
		this.tooltip = this.tooltip || SqueezeJS.string('volumedown');
		SqueezeJS.UI.Buttons.VolumeUp.superclass.initComponent.call(this);
	},

	handler : function(){
		if (this.power)
			SqueezeJS.Controller.setVolume(1, '-');
	}
});

SqueezeJS.UI.Buttons.VolumeUp = Ext.extend(SqueezeJS.UI.Buttons.Base, {
	initComponent : function(){
		this.cls = this.cls || 'btn-volume-increase';
		this.tooltip = this.tooltip || SqueezeJS.string('volumeup');
		SqueezeJS.UI.Buttons.VolumeUp.superclass.initComponent.call(this);
	},

	handler : function(){
		if (this.power)
			SqueezeJS.Controller.setVolume(1, '+');
	}
});

SqueezeJS.UI.VolumeBar = Ext.extend(SqueezeJS.UI.Component, {
	power: null,
	volume : 0,

	initComponent : function(){
		SqueezeJS.UI.VolumeBar.superclass.initComponent.call(this);
	
		if (this.el && (this.el = Ext.get(this.el))) {
			var el;
			if (el = this.el.child('img:first'))
				el.on('click', this.onClick, this);
		}		
	},

	onClick: function(ev, target) {
		if (!this.power)
			return;

		var el = Ext.get(target);
		if (el) {
			var margin = 9;

			var maxWidth = el.getWidth() - 2*margin;
			var myStep = maxWidth/11;

			var myX = ev.xy[0] - el.getX() - margin - (Ext.isGecko * 5) - (Ext.isSafari * 5);
			myX = Math.max(myX, 1);
			myX = Math.min(myX, maxWidth);

			var volVal = Math.ceil(myX / myStep) - 1;

			this.updateState(volVal*10);
			SqueezeJS.Controller.setVolume(volVal);
		}
	},

	// update volume bar
	onPlayerStateChange: function(result){
		if (result['mixer volume'] != null)
			this.updateState(parseInt(result['mixer volume']));

		this.power = result.power;
	},

	updateState: function(newVolume){
		if (newVolume != this.volume) {
			var volEl;
			var volVal = Math.ceil(newVolume / 9.9); 

			if (newVolume <= 0)
				volVal = 0;
			else if (newVolume >= 100)
				volVal = 11;

			this.el.removeClass([ 'ctrlVolume0', 'ctrlVolume1', 'ctrlVolume2', 'ctrlVolume3', 'ctrlVolume4', 'ctrlVolume5', 'ctrlVolume6', 'ctrlVolume7', 'ctrlVolume8', 'ctrlVolume9', 'ctrlVolume10' ]);
			this.el.addClass('ctrlVolume' + String(Math.max(volVal-1, 0)));
	
			if (volEl = this.el.child('img:first'))
				volEl.dom.title = SqueezeJS.string('volume') + ' ' + parseInt(newVolume);

			this.volume = newVolume;
		}
	}
});


SqueezeJS.UI.Title = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.title(result, this.noLink));
	}
});

SqueezeJS.UI.CompoundTitle = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		var title = SqueezeJS.SonginfoParser.title(result, this.noLink, true);
		var contributors = SqueezeJS.SonginfoParser.contributors(result, this.noLink);
		var album = SqueezeJS.SonginfoParser.album(result, this.noLink);

		this.el.update(title
			+ (contributors ? '&nbsp;' + SqueezeJS.string('by') + '&nbsp;' + contributors : '')
			+ (album ? '&nbsp;' + SqueezeJS.string('from') + '&nbsp;' + album : '')
		);
	}
});

SqueezeJS.UI.Album = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		var year = SqueezeJS.SonginfoParser.year(result, this.noLink);
		this.el.update(SqueezeJS.SonginfoParser.album(result, this.noLink)
			+ (year ? '&nbsp;(' + year + ')' : ''));
	}
});

SqueezeJS.UI.Contributors = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.contributors(result, this.noLink));
	}
});


SqueezeJS.UI.CurrentIndex = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(parseInt(result.playlist_cur_index) || 0);
	}
});

SqueezeJS.UI.SongCount = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(parseInt(result.playlist_tracks) + 1);
	}
});

SqueezeJS.UI.Bitrate = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.bitrate(result, this.noLink));
	}
});

SqueezeJS.UI.Playtime = Ext.extend(SqueezeJS.UI.Component, {
	initComponent : function(config){
		if (typeof config == 'string')
			config = { el: config };

		Ext.apply(this, config);
		SqueezeJS.UI.Playtime.superclass.initComponent.call(this);
	
		SqueezeJS.Controller.on({
			'playtimeupdate': {
				fn: this.onPlaytimeUpdate,
				scope: this
			}
		});
	},

	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.current));
	}
});

SqueezeJS.UI.PlaytimeRemaining = Ext.extend(SqueezeJS.UI.Playtime, {
	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.remaining));
	}
});

SqueezeJS.UI.CompoundPlaytime = Ext.extend(SqueezeJS.UI.Playtime, {
	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime)
			this.el.update(SqueezeJS.Utils.formatTime(playtime.current) + '&nbsp;/&nbsp;' + SqueezeJS.Utils.formatTime(playtime.remaining));
	}
});

SqueezeJS.UI.PlaytimeProgress = Ext.extend(SqueezeJS.UI.Playtime, {
	initComponent : function(config){
		SqueezeJS.UI.PlaytimeProgress.superclass.initComponent.call(this);

		var el = Ext.get(this.applyTo);
		el.update( '<img src="/html/images/spacer.gif" class="progressLeft"/><img src="/html/images/spacer.gif" class="progressFillLeft"/>'
			+ '<img src="/html/images/spacer.gif" class="progressIndicator"/><img src="html/images/spacer.gif" class="progressFillRight"/>'
			+ '<img src="/html/images/spacer.gif" class="progressRight"/>' );	

		// store the DOM elements to reduce flicker
		this.remaining = Ext.get(Ext.DomQuery.selectNode('.progressFillRight', el.dom));
		this.playtime = Ext.get(Ext.DomQuery.selectNode('.progressFillLeft', el.dom));
		
		// calculate width of elements which won't be scaled
		this.fixedWidth = el.child('img.progressLeft').getWidth();
		this.fixedWidth += el.child('img.progressRight').getWidth();
		this.fixedWidth += el.child('img.progressIndicator').getWidth();
	},

	onPlaytimeUpdate : function(playtime){
		if (this.el && playtime) {
			var left;
			var max = this.el.getWidth() - this.fixedWidth; // total of left/right/indicator width

			// if we don't know the total play time, just put the indicator in the middle
			if (!playtime.duration)
				left = 0;

			// calculate left/right percentage
			else
				left = Math.max(
						Math.min(
							Math.floor(playtime.current / playtime.duration * max)
						, max)
					, 1);

			this.remaining.setWidth(max - left);
			this.playtime.setWidth(left);
		}
	}
});

SqueezeJS.UI.Coverart = Ext.extend(SqueezeJS.UI.Component, {
	onPlayerStateChange : function(result){
		this.el.update(SqueezeJS.SonginfoParser.coverart(result, this.noLink, this.size));
	}
});

SqueezeJS.UI.CoverartPopup = Ext.extend(Ext.ToolTip, {
	initComponent : function(){
		if (this.songInfo)
			this.title = '&nbsp;';
 
		SqueezeJS.UI.CoverartPopup.superclass.initComponent.call(this);

		SqueezeJS.Controller.on({
			'playerstatechange': {
				fn: this.onPlayerStateChange,
				scope: this
			}
		});
	},

	onPlayerStateChange : function(result){
		if (this.songInfo) {
			var title = SqueezeJS.SonginfoParser.title(result, true, true);
			var contributors = SqueezeJS.SonginfoParser.contributors(result, true);
			var album = SqueezeJS.SonginfoParser.album(result, true);
	
			this.setTitle(title
				+ (contributors ? '&nbsp;/ ' + contributors : '')
				+ (album ? '&nbsp;/ ' + album : ''));
		}

		var el = this.body;
		if (el) {
			if (el = el.child('img:first', true))
				el.src = SqueezeJS.SonginfoParser.coverartUrl(result);
		}
		else {
			this.html = SqueezeJS.SonginfoParser.coverart(result, true);
		}
	}
});